﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/** @file
    @brief      NFP(amiibo) の機能検証に必要な型や定数の宣言
*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/mii/mii_Nfp.h>
#include <nn/nfp/nfp_Types.h>
#include <nn/nfp/nfp_PrivateTypes.h>

namespace nn {
namespace nfp {

/**
 * @brief   アプリケーション専用領域のサイズです。
 */
const int ApplicationAreaSizeV2 = 216;

/**
 * @brief   バックアップデータエントリの総数です。
 */
const int BackupEntryMax = 1000;

/**
 * @brief   バックアップデータのエントリサイズ (bytes) です。
 */
const size_t BackupEntrySize = 2048;

/**
 * @brief         タグの壊し方を指定する定数です。
 */
enum BreakType
{
    BreakType_None,                       //!< タグを壊しません。
    BreakType_Activation,                 //!< 書き込み中情報のフラグを下ろします。
    BreakType_Hmac                        //!< HMAC を書き換えます。
};

/**
 * @brief         NTF の書き込み方を指定する定数です。
 */
enum NtfWriteType
{
    NtfWriteType_Lock,                    //!< 書き込み後ロックします。
    NtfWriteType_NoLock,                  //!< 書き込み後ロックしないため、再度書き込み可能です。
};

/**
 * @brief         NFP タグのシステム管理領域です。
 */
struct SystemInfo
{
    nn::Bit8         writeStatus;         //!< システムがタグに正しく書き込むために操作する書き込み情報です。
    char             _reserved1[1];
    uint16_t         systemWriteCounter;  //!< システム用の書き込み回数カウンタです。
    nn::Bit32        hardwareId;          //!< 最後にタグに書き込みを行った本体の識別子です。
    char             _reserved2[56];
};


/**
 * @brief   オーナー登録情報を示す構造体です。
 */
struct RegisterInfoDebug
{
    nn::mii::NfpStoreDataCore miiDataCore;                    //!< Mii コアデータ
    nn::mii::NfpStoreDataExtention miiDataExtention;          //!< Mii 拡張データ
    Date               registerDate;                          //!< 登録日です。
    char16_t           nickname[NicknameLengthMax + 1];       //!< ニックネーム (+ 終端文字) です。UTF-16 BE で格納されます。
    nn::Bit8           fontRegion;                            //!< ニックネームを表示する際のフォントのリージョン情報です。@ref FontRegion の値が格納されます。
    uint8_t            miiExtentionVersion;                   //!< Mii 拡張データのバージョン
    nn::Bit32          extentionCrc;                          //!< 拡張フォーマット判定用 CRC
    char               reserved[20];                          //!< 拡張フォーマット判定用 CRC の計算に含まれる予約領域
    char               _reserved[100];
};

/**
 * @brief   NFP 共用領域の管理情報を示す構造体です。
 */
struct AdminInfoDebug
{
    nn::ncm::ApplicationId applicationId; //!< 専用領域を作成したアプリケーションの互換用 ID です。
    nn::Bit32      accessId;              //!< 専用領域のアクセス ID です。
    uint16_t       moveCounter;           //!< 本体をまたいで書き込みを行った回数です。
    nn::Bit8       registerInfo;          //!< 各種情報の登録状態を表すビットフィールドです。各ビットは @ref RegisterInfoFlags で定義されます。
    uint8_t        formatVersion;         //!< タグのフォーマットバージョンです。@ref FormatVersion の値を格納してください。
    nn::Bit8       platform;              //!< プラットフォームの種類です。@ref PlatformType の値を格納してください。
    nn::Bit8       applicationIdExt;      //!< 専用領域を作成したアプリケーションの拡張 ID です。
    char           _reserved[46];
};

/**
 * @brief         NFP タグに含まれる全データです。
 */
struct NfpData
{
    SystemInfo        systemInfo;                                //!< ライブラリ用のシステム管理領域です。
    CommonInfo        commonInfo;                                //!< 共有領域です。
    RegisterInfoDebug registerInfo;                              //!< 初期登録情報です。
    AdminInfoDebug    adminInfo;                                 //!< システムアプリ用の共有領域です。
    char              applicationArea[ApplicationAreaSizeV2];    //!< アプリケーション専用領域です。
};

/**
 * @brief         バックアップデータのヘッダを示す構造体です。
 */
struct BackupDataHeader
{
    uint8_t   formatVersion;                 //!< バックアップフォーマットバージョン
    char      _reserved1[1];
    uint16_t  entryNum;                      //!< 使用済みのバックアップデータのエントリ数
    uint16_t  nextEntryIndex;                //!< 次に書き込むエントリの番号
    nn::Bit8  _reserved2[22];
    nn::Bit32 hash;                          //!< ヘッダのハッシュ (CRC32)
};

/**
 * @brief         バックアップデータ用の日付を示す構造体です。
 */
struct BackupDataDate
{
    nn::Bit16 _date;

    /**
     * @brief   西暦を返します。
     *
     * @return  西暦
     */
    int GetYear() const NN_NOEXCEPT
    {
        return (_date >> 9) + 2000;
    }

    /**
     * @brief   月を返します。
     *
     * @return  月(1 から 12 までの値)
     */
    int GetMonth() const NN_NOEXCEPT
    {
        return ((_date >> 5) & 0xF);
    }

    /**
     * @brief   日を返します。
     *
     * @return  日(1 から 31 までの値)
     */
    int GetDay() const NN_NOEXCEPT
    {
        return (_date & 0x1F);
    }

    /**
     * @brief   バックアップデータ用の日付に年月日を設定します。
     *
     * @param[in]   year  西暦
     * @param[in]   month 月(1 から 12 までの値)
     * @param[in]   day   日(1 から 31 までの値)
     */
    void Set(int year, int month, int day) NN_NOEXCEPT
    {
        _date = static_cast<nn::Bit16>((year - 2000) << 9 | month << 5 | day);
    }
};

/**
 * @brief         バックアップデータの目次を示す構造体です。
 */
struct BackupDataToc
{
    uint8_t                 uidLength;                  //!< UID の長さ (1..10)
    nn::Bit8                uid[UidLengthMax];          //!< UID
    char                    _reserved1[1];
    BackupDataDate          entryRegisterDate;          //!< エントリ登録日
    char                    _reserved2[14];
    nn::Bit32               hash;                       //!< 目次のハッシュ (CRC32)
};

/**
 * @brief         バックアップデータを示す構造体です。
 */
struct BackupData
{
    BackupDataHeader header;                                //!< バックアップデータのヘッダ
    BackupDataToc    toc[BackupEntryMax];                   //!< バックアップデータの目次
    char             data[BackupEntryMax][BackupEntrySize]; //!< バックアップデータ
};

}  // nfp
}  // nn
