﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <nn/os/os_Thread.h>
#include <nn/os/os_ReaderWriterLock.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/hid/hid_NpadCommonTypes.h>
#include <nn/nfc/server/core/nfc_Result.h>
#include <nn/nfc/server/core/nfc_Types.h>
#include <nn/nfc/server/core/nfc_CoreService.h>
#include <nn/nfp/detail/nfp_Port.h>
#include <nn/nfc/detail/nfc_Port.h>
#include <nn/nfc/mifare/detail/nfc_MifarePort.h>
#include <nn/nfc/server/util/nfc_ThreadStack.h>

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
namespace nn { namespace psc {
    class PmModule;
}}
#endif

namespace nn { namespace nfc { namespace server { namespace core {
    class Device;

    class Manager
    {
    public:
        void Initialize(Service* service) NN_NOEXCEPT;
        void Finalize(Service* service) NN_NOEXCEPT;
        nn::Result ListDevices(uint64_t *pOutDeviceUniqueId, nn::nfc::DeviceHandle* pOutBuffer, int* pOutCount, Service* service, int bufferCount) NN_NOEXCEPT;
        nn::Result StartDiscovery(Service* service, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT;
        nn::Result StartDiscovery(Service* service, nn::nfc::DeviceHandle deviceHandle, nn::nfc::NfcProtocol protocolFilter) NN_NOEXCEPT;
        nn::Result StopDiscovery(Service* service, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT;
        nn::Result KeepSession(Service* service, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT;
        nn::Result ReleaseSession(Service* service, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT;
        void SetEvent(nn::os::SystemEventType* pLinkEvent) NN_NOEXCEPT;
        nn::Result SetEvent(nn::os::SystemEventType* pEvent, nn::os::SystemEventType* pDetectEvent, nn::os::SystemEventType* pResetEvent, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT;
        nn::Result GetInfo(Info* pInfo, Service* service, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT;
        nn::Result GetDetectInfo(Info* pInfo, Service* service, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT;
        nn::Result GetResetInfo(ResetInfo* pResetInfo, Service* service, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT;
        nn::Result StartNtagRead(Service* service, nn::nfc::DeviceHandle deviceHandle, const nn::nfc::TagId& tagId, const NtagReadParameter& ntagReadParameter) NN_NOEXCEPT;
        nn::Result StartNtagWrite(Service* service, nn::nfc::DeviceHandle deviceHandle, const nn::nfc::TagId& tagId, const NtagWriteParameter& ntagWriteParameter) NN_NOEXCEPT;
        nn::Result SendRawData(Service* service, nn::nfc::DeviceHandle deviceHandle, const nn::nfc::TagId& tagId, const PassThruParameter& passThruParameter) NN_NOEXCEPT;
        nn::Result GetTagInfo(nn::nfc::TagInfo* tagInfo, Service* service, nn::nfc::DeviceHandle deviceHandle, const nn::nfc::TagId& tagId) NN_NOEXCEPT;
        nn::Result GetNpadId(nn::hid::NpadIdType* pOutNpadId, Service* service, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT;
        nn::Result GetXcdDeviceHandle(nn::xcd::DeviceHandle* pOutXcdDeviceHandle, Service* service, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT;
        nn::Result IsNfcEnabled(bool* pOutIsNfcEnabled, Service* service) NN_NOEXCEPT;
        nn::Result SetNfcEnabled(Service* service, bool isNfcEnabled) NN_NOEXCEPT;
        nn::Result CheckService(Service* service) NN_NOEXCEPT;
        nn::Result CheckNfcEnabled(Service* service) NN_NOEXCEPT;
        nn::Result CheckAwake() NN_NOEXCEPT;
        nn::Result NotifyForegroundApplet(nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;
        nn::Bit64 GetTerminalId() NN_NOEXCEPT;
        nn::Result StartMifareRead(Service* service, nn::nfc::DeviceHandle deviceHandle, const nn::nfc::TagId& tagId, const MifareReadParameter& mifareReadParameter) NN_NOEXCEPT;
        nn::Result StartMifareWrite(Service* service, nn::nfc::DeviceHandle deviceHandle, const nn::nfc::TagId& tagId, const MifareWriteParameter& mifareWriteParameter) NN_NOEXCEPT;
        void SignalAvailabilityChangeEvent() NN_NOEXCEPT;
        void SignalEvent(Service* service, nn::nfc::DeviceHandle deviceHandle, bool isActivate) NN_NOEXCEPT;
        bool IsInitialized(nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

        // 現在のスレッドが LinkMonitoringThread かどうか
        bool IsInLinkMonitoringThread() NN_NOEXCEPT
        {
            return nn::os::GetCurrentThread() == &m_LinkMonitoringThread;
        }

    private:
        Manager() NN_NOEXCEPT;
        ~Manager() NN_NOEXCEPT;
        nn::Result CheckServicePrivate(Service* service) NN_NOEXCEPT;
        nn::Result CheckNfcEnabledPrivate(Service* service) NN_NOEXCEPT;
        nn::Result CheckAwakePrivate() NN_NOEXCEPT;
        nn::Result GetDevice(Device** pOutDevice, Service* service, nn::nfc::DeviceHandle deviceHandle, bool check) NN_NOEXCEPT;
        nn::Result GetDevice(Device** pOutDevice, Service* service, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT;
        static void LinkMonitoringThread(void* arg) NN_NOEXCEPT;
        void UpdateDevices() NN_NOEXCEPT;
        bool IsInitialized() NN_NOEXCEPT;
        void InitializeTerminalId() NN_NOEXCEPT;
        bool IsNfcEnabled() NN_NOEXCEPT;
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        void PmHandler() NN_NOEXCEPT;
#endif
        void ResetDevices(ResetReason reason, Service* service) NN_NOEXCEPT;
        void SignalLinkEvent() NN_NOEXCEPT;

    public:
        static Manager& GetInstance() NN_NOEXCEPT
        {
            static Manager instance;
            return instance;
        }

    private:
        uint64_t m_DeviceUniqueId;
        int m_InitializedCount;
        std::unique_ptr<Device> m_DeviceList[nn::nfc::DeviceCountMax];

        nn::os::SystemEventType m_LinkEvent;
        nn::os::MultiWaitHolderType m_LinkHolder;
        nn::os::MultiWaitType m_LinkMonitoringMultiWait;
        nn::os::SystemEventType m_TerminateLinkMonitoringEvent;
        nn::os::MultiWaitHolderType m_TerminateLinkMonitoringHolder;
        nn::os::ThreadType m_LinkMonitoringThread;
        std::unique_ptr<nn::nfc::server::util::ThreadStack> m_LinkMonitoringThreadStack;

        mutable nn::os::ReaderWriterLock m_ReaderWriterLock;
        bool m_IsRunningThread;
        static const int ServiceCountMax = nn::nfp::detail::UserSessionCount + nn::nfp::detail::SystemSessionCount + nn::nfc::mifare::detail::UserSessionCount + nn::nfp::detail::DebugSessionCount;
        Service* m_Service[ServiceCountMax];
        nn::applet::AppletResourceUserId m_ForegroundApplet;
        nn::os::SystemEventType* m_pLinkEventForNfcUser;
        bool m_IsInitializedTerminalId;
        nn::Bit64 m_TerminalId;
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        std::unique_ptr<nn::psc::PmModule> m_PmModule;
#endif
        nn::os::MultiWaitHolderType m_PmModuleHolder;
        bool m_IsAwake;
    };



}}}}  // namespace nn::nfc::server::core

