﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/news/detail/service/news_Common.h>

namespace nn { namespace news { namespace detail { namespace service { namespace core {

/*!
    @brief      受信履歴管理モジュールです。
*/
class ReceivedHistoryManager
{
public:
    /*!
        @brief      受信履歴の最大数です。
    */
    static const int HistoryCountMax = 12800;

    /*!
        @brief      ブロック単位の読み込み件数です。

        @details
                    本クラスは、ブロック単位で読み込みや削除（空き確保）を行います。
    */
    static const int CountPerBlock = 1280;

    /*!
        @brief      受信履歴の検索用キーです。
    */
    struct SearchKey
    {
        //
        bool isIndividual;
        uint64_t userId;
        //
        char domain;
        uint64_t newsId;
        //
        int8_t language;
        uint64_t dataId;
    };

    /*!
        @brief      受信履歴レコードです。
    */
    struct Record
    {
        // 8
        bool isIndividual;
        int8_t language;
        char domain;
        char reserved[5];
        // 8
        uint64_t userId;
        // 8
        uint64_t newsId;
        // 8
        uint64_t dataId;
        // 8
        nn::time::PosixTime receivedTime;
    };

private:
    /*!
        @brief      コンストラクタです。
    */
    ReceivedHistoryManager() NN_NOEXCEPT;

public:
    /*!
        @brief      インスタンスを取得します。

        @return     インスタンス。
    */
    static ReceivedHistoryManager& GetInstance() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(ReceivedHistoryManager, instance);
        return instance;
    }

public:
    /*!
        @brief      受信履歴を追加、または、更新します。

        @param[in]  record  受信履歴。

        @return     処理結果。

        @details
                    指定した主キーのレコードが存在する場合、更新されます。

                    受信履歴ファイルに空きが存在しない場合、前方の @ref ReceivedHistoryManager::CountPerBlock 件のレコードを削除します。
    */
    nn::Result AddOrUpdate(const Record& record) NN_NOEXCEPT;

    /*!
        @brief      受信履歴をクリアします。

        @return     処理結果。
    */
    nn::Result Clear() NN_NOEXCEPT;

    /*!
        @brief      ニュースがインポート可能かどうかを確認します。

        @param[out] outIsImportable インポート可能かどうか。
        @param[in]  key             検索キー。

        @return     処理結果。

        @pre
            - outIsImportable != nullptr
    */
    nn::Result IsImportable(bool* outIsImportable, const SearchKey& key) NN_NOEXCEPT;

    /*!
        @brief      ニュースが上書き可能かどうかを確認します。

        @param[out] outIsOverwritable   上書き可能かどうか。
        @param[in]  key                 検索キー。
        @param[in]  dataId              データ ID。

        @return     処理結果。

        @pre
            - outIsOverwritable != nullptr
    */
    nn::Result IsOverwritable(bool* outIsOverwritable, const SearchKey& key) NN_NOEXCEPT;

private:
    //
    nn::os::Mutex m_Mutex;
    //
    Record m_Records[CountPerBlock];
    int m_Count;
    int m_BlockIndex;

private:
    //
    nn::Result Open(size_t* outSize, nn::fs::FileHandle* outHandle) NN_NOEXCEPT;
    //
    nn::Result Create(nn::fs::FileHandle* outHandle) NN_NOEXCEPT;
    //
    nn::Result Read(nn::fs::FileHandle handle, int blockIndex) NN_NOEXCEPT;
    //
    nn::Result DeleteHeadBlock(nn::fs::FileHandle handle) NN_NOEXCEPT;
    //
    nn::Result SearchUpdatableRecord(int* outIndex, nn::fs::FileHandle handle, size_t size, const Record& record) NN_NOEXCEPT;
    int SearchUpdatableRecordFromCache(const Record& record) NN_NOEXCEPT;
    //
    int SearchRecordForImport(const SearchKey& key) NN_NOEXCEPT;
    int SearchRecordForOverwrite(const SearchKey& key) NN_NOEXCEPT;
};

}}}}}
