﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/news/detail/service/news_Common.h>
#include <sqlite3.h>

namespace nn { namespace news { namespace detail { namespace service { namespace core {

/*!
    @brief      ニュースデータベースです。
*/
class NewsDatabase
{
private:
    NN_DISALLOW_COPY(NewsDatabase);
    NN_DISALLOW_MOVE(NewsDatabase);

public:
    /*!
        @brief      ニュースデータベースに登録するアプリケーション ID の最大数です。
    */
    static const int ApplicationIdCountMax = 5;

    /*!
        @brief      ニュースデータベースに挿入するレコードです。
    */
    struct InsertRecord
    {
        NewsId newsId;
        UserId userId;
        nn::time::PosixTime receivedAt;
        nn::time::PosixTime publishedAt;
        nn::time::PosixTime expireAt;
        nn::time::PosixTime pickupLimit;
        int32_t priority;
        int32_t deletionPriority;
        int32_t ageLimit;
        nn::ApplicationId applicationIds[ApplicationIdCountMax];
        TopicId topicId;
        int32_t surprise;
        int32_t bashotorya;
        int32_t point;
        int32_t read;
        int32_t newly;
        int32_t displayed;
        int32_t optedIn;
        int32_t pointStatus;
        int32_t extra1;
        int32_t extra2;
    };

private:
    /*!
        @brief      コンストラクタです。
    */
    NewsDatabase() NN_NOEXCEPT;

public:
    /*!
        @brief      インスタンスを取得します。

        @return     インスタンス。
    */
    static NewsDatabase& GetInstance() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(NewsDatabase, s_Instance);
        return s_Instance;
    }

public:
    /*!
        @brief      指定したクエリでレコードを取得します。

        @param[out] outCount        取得した数。
        @param[out] outRecords      レコードリスト。
        @param[in]  wherePhrase     WHERE 句。
        @param[in]  orderByPhrase   ORDER BY 句。
        @param[in]  offset          オフセット。
        @param[in]  count           レコードリストの要素数。

        @return     処理結果。

        @pre
            - outCount != nullptr
            - outRecords != nullptr
            - wherePhrase != nullptr
            - orderByPhrase != nullptr
            - offset >= 0
            - count > 0
            - システム情報ストレージをマウントしている。

        @details
                    本関数は、以下の SQL 文を発行してフィルタリング・ソートを行います。@n
                    WHERE 句を空文字列にした場合、全てのレコードが取得できます。

                    - SELECT * FROM news WHERE $wherePhrase ORDER BY $orderByPhrase LIMIT $offset, $count
    */
    nn::Result GetList(int* outCount, NewsRecord* outRecords,
        const char* wherePhrase, const char* orderByPhrase, int offset, int count) NN_NOEXCEPT;

    /*!
        @brief      指定したクエリでレコードを取得します。

        @param[out] outCount        取得した数。
        @param[out] outRecords      レコードリスト。
        @param[in]  wherePhrase     WHERE 句。
        @param[in]  orderByPhrase   ORDER BY 句。
        @param[in]  offset          オフセット。
        @param[in]  count           レコードリストの要素数。

        @return     処理結果。

        @pre
            - outCount != nullptr
            - outRecords != nullptr
            - wherePhrase != nullptr
            - orderByPhrase != nullptr
            - offset >= 0
            - count > 0
            - システム情報ストレージをマウントしている。

        @details
                    本関数は、以下の SQL 文を発行してフィルタリング・ソートを行います。@n
                    WHERE 句を空文字列にした場合、全てのレコードが取得できます。

                    - SELECT * FROM news WHERE $wherePhrase ORDER BY $orderByPhrase LIMIT $offset, $count
    */
    nn::Result GetList(int* outCount, NewsRecordV1* outRecords,
        const char* wherePhrase, const char* orderByPhrase, int offset, int count) NN_NOEXCEPT;

    /*!
        @brief      指定したクエリでデータ件数を取得します。

        @param[out] outCount    取得した数。
        @param[in]  wherePhrase WHERE 句。

        @return     処理結果。

        @pre
            - outCount != nullptr
            - wherePhrase != nullptr

        @details
                    本関数は、以下の SQL 文を発行して件数を取得します。@n
                    WHERE 句を空文字列にした場合、保存されているデータ件数を取得できます。

                    - SELECT COUNT(*) FROM news WHERE $wherePhrase
    */
    nn::Result Count(int* outCount, const char* wherePhrase) NN_NOEXCEPT;

    /*!
        @brief      指定したクエリでデータ件数を取得します。

        @param[out] outCount    取得した数。
        @param[in]  distinct    キーの重複を排除するかどうか。@n
                                キーの重複を排除した場合、指定したキーのユニーク数を取得することができます。
        @param[in]  key         キー。
        @param[in]  wherePhrase WHERE 句。

        @return     処理結果。

        @pre
            - outCount != nullptr
            - key != nullptr
            - wherePhrase != nullptr
            - システム情報ストレージをマウントしている。

        @details
                    本関数は、以下の SQL 文を発行して件数を取得します。

                    - SELECT COUNT([DISTINCT] $key) FROM news WHERE $wherePhrase
    */
    nn::Result Count(int* outCount, bool distinct, const char* key, const char* wherePhrase) NN_NOEXCEPT;

    /*!
        @brief      指定した条件に当てはまる整数値を持つキーの値を更新します。

        @param[in]  key         キー。
        @param[in]  newValue    新しい値。
        @param[in]  wherePhrase WHERE 句。

        @return     処理結果。

        @pre
            - key != nullptr
            - wherePhrase != nullptr
            - システム情報ストレージをマウントしている。

        @details
                    指定したキーが整数値を持たない場合や更新不可の場合、本関数は失敗します。

                    本関数は以下の SQL 文を発行します。

                    - UPDATE news SET $key = $newValue WHERE $wherePhrase
    */
    nn::Result Update(const char* key, int32_t newValue, const char* wherePhrase) NN_NOEXCEPT;

    /*!
        @brief      指定した条件に当てはまるキーの文字列値を更新します。

        @param[in]  key         キー。
        @param[in]  newValue    新しい値。
        @param[in]  wherePhrase WHERE 句。

        @return     処理結果。

        @pre
            - key != nullptr
            - newValue != nullptr
            - wherePhrase != nullptr
            - システム情報ストレージをマウントしている。

        @details
                    指定したキーが文字列値を持たない場合や更新不可の場合、本関数は失敗します。

                    本関数は以下の SQL 文を発行します。

                    - UPDATE news SET $key = '$newValue' WHERE $wherePhrase
    */
    nn::Result Update(const char* key, const char* newValue, const char* wherePhrase) NN_NOEXCEPT;

    /*!
        @brief      指定した条件に当てはまる整数値を持つキーの値を加算更新します。

        @param[in]  key         キー。
        @param[in]  value       加算値。
        @param[in]  wherePhrase WHERE 句。

        @return     処理結果。

        @pre
            - key != nullptr
            - wherePhrase != nullptr
            - システム情報ストレージをマウントしている。

        @details
                    指定したキーが整数値を持たない場合や更新不可の場合、本関数は失敗します。

                    本関数は以下の SQL 文を発行します。

                    - UPDATE news SET $key = $key + $value WHERE $wherePhrase
    */
    nn::Result UpdateWithAddition(const char* key, int32_t value, const char* wherePhrase) NN_NOEXCEPT;

    /*!
        @brief      追加済みのレコードを更新します。

        @param[in]  record  レコード。

        @return     処理結果。

        @pre
            - システム情報ストレージをマウントしている。

        @details
                    以下のカラムは更新しません。

                    - topic_id
                    - read
                    - newly
                    - displayed
    */
    nn::Result UpdateInsertedRecord(const InsertRecord& record) NN_NOEXCEPT;

    /*!
        @brief      レコードを追加します。

        @param[in]  record  レコード。

        @return     処理結果。

        @pre
            - システム情報ストレージをマウントしている。
    */
    nn::Result Insert(const InsertRecord& record) NN_NOEXCEPT;

    /*!
        @brief      指定したクエリでレコードを削除します。

        @param[in]  wherePhrase WHERE 句。

        @return     処理結果。

        @pre
            - wherePhrase != nullptr
            - システム情報ストレージをマウントしている。
    */
    nn::Result Delete(const char* wherePhrase) NN_NOEXCEPT;

    /*!
        @brief      データベースを削除します。

        @return     処理結果。

        @pre
            - システム情報ストレージをマウントしている。
    */
    nn::Result DeleteAll() NN_NOEXCEPT;

    /*!
        @brief      データベースを縮小します。

        @param[out] outIsCommitRequired コミットが必要かどうか。

        @return     処理結果。

        @pre
            - outIsCommitRequired != nullptr
            - システム情報ストレージをマウントしている。

        @details
                    データベースのページサイズが大きい場合、適正サイズに変更して再生成します。
    */
    nn::Result Shrink(bool* outIsCommitRequired) NN_NOEXCEPT;

    /*!
        @brief      データベースのダンプを取得します。

        @param[out] outSize 読み込んだサイズ。
        @param[in]  buffer  読み込みバッファ。
        @param[in]  size    読み込みバッファのサイズ。

        @return     処理結果。

        @pre
            - システム情報ストレージをマウントしている。
    */
    nn::Result GetDump(size_t* outSize, void* buffer, size_t size) NN_NOEXCEPT;

private:
    //
    nn::os::Mutex m_Mutex;

private:
    //
    nn::Result Open(sqlite3** outHandle) NN_NOEXCEPT;
    void Close(sqlite3* pHandle) NN_NOEXCEPT;
    //
    nn::Result GetPageSize(int* outPageSize) NN_NOEXCEPT;
};

}}}}}
