﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

/**
 *  @file
 *  @brief Mii画像データベースアクセス定義です。
 *  @details Mii画像データベースのアクセサを定義します。
 */

#include <nn/nn_Result.h>
#include <nn/util/util_TypedStorage.h>
#include <nn/util/util_UuidTypes.h>
#include <nn/mii/mii_Common.h>
#include <nn/mii/mii_PrivateCommon.h>

namespace nn { namespace mii {

/**
 * @brief Mii画像の属性情報構造体
 * @details 1つのMii画像の属性を表す構造体です。
 */
struct ImageAttribute
{
    nn::util::Uuid         id;         ///< 画像の識別Id
    nn::mii::CreateId      createId;   ///< Mii画像に使ったMiiのId
    nn::mii::Expression    expression; ///< 表情
    nn::mii::ImagePoseType poseType;   ///< ポーズ種類
    nn::mii::FontRegion    fontRegion; ///< フォントのリージョン
    nn::mii::Nickname      nickname;   ///< ニックネーム
};

/**
 * @brief Mii画像データベース
 * @details Mii画像にアクセスするためのインターフェイスです。
 */
class ImageDatabase
{
    NN_DISALLOW_COPY(ImageDatabase) NN_NOEXCEPT;
    bool operator ==(const ImageDatabase& rhs) const NN_NOEXCEPT; ///< 比較禁止
    bool operator !=(const ImageDatabase& rhs) const NN_NOEXCEPT; ///< 比較禁止
public:
    ///! 画像の幅
    static const int ImageWidth  = 256;
    ///! 画像の高さ
    static const int ImageHeight = ImageWidth;
    ///! 1枚あたりの画像サイズ
    static const int ImageSize = ImageWidth * ImageHeight * 4;
    ///! データベースに格納できる画像の枚数
    static const int MaxCount = 9;
    ///! データベースファイルのサイズ
    static const int MaxFileSize = 2359804;

    ImageDatabase() NN_NOEXCEPT;
    ~ImageDatabase() NN_NOEXCEPT;

    /**
     * @brief データベースを初期化します。
     * @param[out] pResult      データベース読み込み結果を返します。
     * @param[in]  enableFormat データベースに破損があった場合にフォーマットするか
     * @return     処理の結果が返ります。
     * @retval     ResultSuccess           成功しました。
     * @retval     nn::mii::ResultBroken   データベースが破損しています(enableFormat=falseの場合に返ります)
     * @pre pResult != nullptr
     * @details ヘッダーを読み込みデータベースを初期化します。
     *          pResult は、戻り値が ResultSuccess の時のみ有効です。
     */
    nn::Result Initialize(DatabaseResult* pResult,bool enableFormat) NN_NOEXCEPT;

    /**
     * @brief データベースが初期化済みか判定します。
     * @return 初期化済みならtrueを返します。
     * @details
     */
    bool IsInitialized() const NN_NOEXCEPT;

    /**
     * @brief データベースのリロードを行います。
     * @return     処理の結果が返ります。
     * @retval     ResultSuccess           成功しました。
     * @pre IsInitialized() == true
     * @details ヘッダーの再読込をこないます。
     *          他のプロセスでの更新を反映します。
     */
    nn::Result Reload() NN_NOEXCEPT;

    /**
     * @brief データベース内の有効なMii画像の数を返します。
     * @return 有効な要素数を返します。
     * @pre IsInitialized() == true
     * @details
     */
    int GetCount() const NN_NOEXCEPT;

    /**
     * @brief データベースが空か判定します。
     * @return 空ならtrueを返します。
     * @pre IsInitialized() == true
     * @details
     */
    bool IsEmpty() const NN_NOEXCEPT;

    /**
     * @brief データベースが満杯か判定します。
     * @return 満杯ならtrueを返します。
     * @pre IsInitialized() == true
     * @details
     */
    bool IsFull() const NN_NOEXCEPT;

    /**
     * @brief 指定したindexの属性を取得します。
     * @param[out] pAttribute 属性の出力先
     * @param[in]  index      データベースのindex
     * @pre IsInitialized() == true
     *      && index < GetCount()
     *      && pAttribute != nullptr
     * @details
     */
    void GetAttribute(ImageAttribute* pAttribute, int index) const NN_NOEXCEPT;

    /**
     * @brief 指定(id)した画像をバッファにロードします。
     * @param[out] pOut 出力先バッファ
     * @param[in]  size pOutのbyteサイズ
     * @param[in]  id   取得する画像のuuid
     * @return 取得に成功するとnn::ResultSuccessを返します。
     * @pre IsInitialized() == true
     *      && pOut != nullptr
     *      && size == ImageSize
     *      && id != nn::util::InvalidUuid
     * @details
     */
    nn::Result LoadImage(void* pOut, size_t size, const nn::util::Uuid& id) const NN_NOEXCEPT;

    /**
     * @brief 指定した画像の追加もしくは更新を行う
     * @param[in] pOutIndex           Mii画像データベース内の書き込み先index番号
     * @param[in] expression          Mii画像の表情
     * @param[in] pose                Mii画像のポーズ
     * @param[in] fontRegion          nicknameのFontRegion
     * @param[in] nickname            Mii画像を作成したときに利用したMiiのニックネーム
     * @param[in] createId            MiiのCreateId
     * @param[in] pUsedUuid           更新するMii画像Uuid(追加の場合はnullptrを指定します)
     * @param[in] pValidUuidArray     削除してはいけないMii画像Id配列
     * @param[in] validUuidArrayCount pValidUuidArrayの要素数
     * @param[in] pSrc 書き込む画像のバッファ
     * @param[in] size pSrcのサイズ
     * @return 書き込みに成功するとnn::ResultSuccessを返します。
     * @pre IsInitialized() == true
     *      && IsFull() != false
     * @details pOutIndexは、nn::ResultSuccess()の場合のみ有効です。
     *          内部でデータベースファイルの書き込みとCommitを行います。
     *          pUsedUuid == nullptr の場合は、追加として扱われます。
     *          pUsedUuid != nullptr の場合は、更新として扱われます。
     *          PSELとの連携の都合で
     *          更新の場合は、新しいUuidが付与されたMii画像がpOutIndexの場所に保存されます。
     *          詳細については、 MIIEDITN-255 を参照してください。
     */
    nn::Result AddOrUpdateImage(
        int* pOutIndex,
        Expression expression,
        ImagePoseType pose,
        FontRegion fontRegion,
        const Nickname& nickname,
        const nn::mii::CreateId& createId,
        const nn::util::Uuid* pUsedUuid,
        const nn::util::Uuid* pValidUuidArray,
        int validUuidArrayCount,
        const void* pSrc,
        int size) NN_NOEXCEPT;

    /**
     * @brief 指定した画像を複数枚同時に削除します。
     * @param[in] pIds  削除するIdの配列
     * @param[in] count pIds配列の要素数
     * @return 削除に成功するとnn::ResultSuccessを返します。
     * @pre IsInitialized() == true
     *      && count >= 1
     *      && count <= MaxCount
     * @details pIdsで指定したIdをcount分削除します。
     *          内部でデータベースファイルの書き込みとCommitを行います。
     *          複数同時に削除する場合はこちらのAPIを使用してください。
     *          ファイルIOが発生する関係でこちらの方が高速です。
     */
    nn::Result DeleteImages(const nn::util::Uuid* pIds,int count) NN_NOEXCEPT;


    //! @name デバッグ用
    //! @{

    /**
     * @brief データベースファイルを削除します。
     * @pre Mii画像データベースのテストモードが有効
     * @details Mii画像データベースのファイルを削除します。
     *          処理に失敗するとABORTします。
     */
    static void DeleteFile() NN_NOEXCEPT;

    /**
     * @brief データベースファイルを破壊します。
     * @pre Mii画像データベースのテストモードが有効
     * @details ファイルが存在しない場合は、作成して破壊します。
     *          処理に失敗するとABORTします。
     */
    static void DestroyFile() NN_NOEXCEPT;

    /**
     * @brief データベースファイルをインポートします。
     * @param[in] pSrc インポートファイルバッファ
     * @param[in] size pSrcのバッファサイズ
     * @pre Mii画像データベースのテストモードが有効
     * @details 処理に失敗するとABORTします。
     */
    static void ImportFile(const void* pSrc,size_t size) NN_NOEXCEPT;

    /**
     * @brief データベースファイルをエクスポートします。
     * @param[out] pOut 書き込み先バッファ
     * @param[in]  size pOutのバッファサイズ(@ref MaxFileSize)
     * @return ファイルのサイズ
     * @pre Mii画像データベースのテストモードが有効
     * @details 処理に失敗するとABORTします。
     */
    static size_t ExportFile(void* pOut,size_t size) NN_NOEXCEPT;

    //! @}

private:
    bool m_IsInitialized;
};

}}
