﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <nn/lidbe/lidbe_Types.h>

namespace nn { namespace lidbe {

/*!
    @brief         初期化します。

                   内部で SSL コンテキストの作成を行います。@n
                   そのため、本関数を呼び出す前に nn::socket::Initialize() / curl_global_init() を呼び出しておく必要があります。

    @return        初期化が成功すれば true、失敗すれば false が返ります。
*/
bool Initialize();

/*!
    @brief         終了します。

                   内部で作成した SSL コンテキストを破棄します。
*/
void Finalize();


/*!
    @brief         アイコンファイルをダウンロードします。

                   本関数を呼び出す前に nn::nifm::SubmitNetworkRequestAndWait() でネットワークとの接続を確立してください。

                   本関数は内部で curl_easy_perform() を呼び出します。@n
                   そのため、処理がブロックされてしまうことに注意してください。

    @param[out]    pFileBuffer     アイコンファイルを格納するバッファ。@n
                                   WUP タイトルの場合、バッファのサイズは IDBE_FILE_SIZE_WUP 以上にしてください。@n
                                   CTR タイトルの場合、バッファのサイズは IDBE_FILE_SIZE_CTR 以上にしてください。
    @param[in]     programId       タイトル ID。
    @param[in]     version         バージョン。
    @param[in]     isKeepAlive     Keep-Alive を行うかどうか。

    @return        ダウンロードが成功すれば true、失敗すれば false が返ります。
*/
bool DownloadIconFile(void* pFileBuffer, uint64_t programId, uint16_t version, bool isKeepAlive);

/*!
    @brief         最新バージョンのアイコンファイルをダウンロードします。

                   本関数を呼び出す前に nn::nifm::SubmitNetworkRequestAndWait() でネットワークとの接続を確立してください。

                   本関数は内部で curl_easy_perform() を呼び出します。@n
                   そのため、処理がブロックされてしまうことに注意してください。

    @param[out]    pFileBuffer     アイコンファイルを格納するバッファ。@n
                                   WUP タイトルの場合、バッファのサイズは IDBE_FILE_SIZE_WUP 以上にしてください。@n
                                   CTR タイトルの場合、バッファのサイズは IDBE_FILE_SIZE_CTR 以上にしてください。
    @param[in]     programId       タイトル ID。
    @param[in]     isKeepAlive     Keep-Alive を行うかどうか。

    @return        ダウンロードが成功すれば true、失敗すれば false が返ります。
*/
bool DownloadLatestIconFile(void* pFileBuffer, uint64_t programId, bool isKeepAlive);

/*!
    @brief         アイコンファイルのダウンロードをキャンセルします。

                   本関数は、@ref DownloadIconFile または @ref DownloadLatestIconFile を使用してアイコンファイルをダウンロードする時のみ使用可能です。@n
                   上記の関数を使用するスレッドはブロックされるため、別スレッドから呼び出してください。

                   ダウンロードコンテキストを作成して自前でアイコンをダウンロードする場合は使用しないでください。
*/
void Cancel();

/*!
    @brief         ダウンロードコンテキストを作成します。

                   本関数を呼び出す前に curl_global_init() を呼び出しておく必要があります。

                   本関数はアイコンファイルのダウンロードに必要なコンテキストを作成します。@n
                   ダウンロード処理は DownloadContext::pCurl を使用して curl_easy_perform() / curl_multi_perform() してください。

                   ダウンロード完了後、ダウンロード結果は @ref CheckDownloadResult() でチェックします。@n
                   その後、 @ref DestroyDownloadContext() でダウンロードコンテキストを破棄してください。

    @param[out]    pContext        ダウンロードコンテキスト。
    @param[out]    pFileBuffer     アイコンファイルを格納するバッファ。@n
                                   WUP タイトルの場合、バッファのサイズは IDBE_FILE_SIZE_WUP 以上にしてください。@n
                                   CTR タイトルの場合、バッファのサイズは IDBE_FILE_SIZE_CTR 以上にしてください。
    @param[in]     programId       タイトル ID。
    @param[in]     version         バージョン。
    @param[in]     isKeepAlive     Keep-Alive を行うかどうか。

    @return        作成が成功すれば true、失敗すれば false が返ります。
*/
bool CreateIconDownloadContext(DownloadContext* pContext, void* pFileBuffer, uint64_t programId, uint16_t version, bool isKeepAlive);

/*!
    @brief         ダウンロードコンテキストを作成します。

                   本関数を呼び出す前に curl_global_init() を呼び出しておく必要があります。

                   本関数は最新バージョンのアイコンファイルのダウンロードに必要なコンテキストを作成します。@n
                   ダウンロード処理は DownloadContext::pCurl を使用して curl_easy_perform() / curl_multi_perform() してください。

                   ダウンロード完了後、ダウンロード結果は @ref CheckDownloadResult() でチェックします。@n
                   その後、 @ref DestroyDownloadContext() でダウンロードコンテキストを破棄してください。

    @param[out]    pContext        ダウンロードコンテキスト。
    @param[out]    pFileBuffer     アイコンファイルを格納するバッファ。@n
                                   WUP タイトルの場合、バッファのサイズは IDBE_FILE_SIZE_WUP 以上にしてください。@n
                                   CTR タイトルの場合、バッファのサイズは IDBE_FILE_SIZE_CTR 以上にしてください。
    @param[in]     programId       タイトル ID。
    @param[in]     isKeepAlive     Keep-Alive を行うかどうか。

    @return        作成が成功すれば true、失敗すれば false が返ります。
*/
bool CreateLatestIconDownloadContext(DownloadContext* pContext, void* pFileBuffer, uint64_t programId, bool isKeepAlive);

/*!
    @brief         ダウンロードコンテキストを破棄します。

    @param[in]     pContext        ダウンロードコンテキスト。
*/
void DestroyDownloadContext(DownloadContext* pContext);

/*!
    @brief         ダウンロード処理の結果をチェックします。

                   本関数はダウンロード処理後に呼び出してください。

    @param[in]     pContext        ダウンロードコンテキスト。

    @return        ダウンロードが成功していれば true、失敗していれば false が返ります。
*/
bool CheckDownloadResult(DownloadContext* pContext);

/*!
    @brief         アイコンファイルの復号化処理を行います。

    @param[out]    pIconBuffer     アイコンデータを格納するバッファ。@n
                                   WUP タイトルの場合、バッファのサイズは IDBE_SIZE_WUP 以上にしてください。@n
                                   CTR タイトルの場合、バッファのサイズは IDBE_SIZE_CTR 以上にしてください。
    @param[in]     pFileBuffer     ダウンロードしたアイコンファイルのバッファ。@n
                                   WUP タイトルの場合、バッファのサイズは IDBE_FILE_SIZE_WUP 以上にしてください。@n
                                   CTR タイトルの場合、バッファのサイズは IDBE_FILE_SIZE_CTR 以上にしてください。
    @param[in]     programId       タイトル ID。

    @return        復号化処理が成功すれば true、失敗すれば false が返ります。
*/
bool DecryptIconFile(void* pIconBuffer, const void* pFileBuffer, uint64_t programId);


namespace detail
{
    nn::Result GetLastResult();
}

}}
