﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/jit/jit_CommonTypes.h>

namespace nn { namespace jit { namespace plugin {

/**
    @brief Jit プラグインの環境を格納する構造体です。

    @details
     本構造体のうち有効な値を持つメンバーは、本構造体を引数にとる関数によって異なります。
*/
struct JitPluginEnvironment
{
    JitEnvironmentInfo environmentInfo;
    void* workMemory;
    size_t workMemorySize;
    void* rxWritable;
    void* roWritable;
};

/**
    @brief プラグインロード時に呼ばれる関数です。
*/
void ConfigurePlugin() NN_NOEXCEPT;

/**
    @brief プラグイン開始時に呼ばれる関数です。
*/
void OnPrepared(const nn::jit::plugin::JitPluginEnvironment& info) NN_NOEXCEPT;

/**
    @brief nn::jit::Control() が呼ばれた際に呼ばれる関数です。

    @pre info.rxWritable == nullptr
    @pre info.roWritable == nullptr

    @details
     本関数呼び出し時にはコード領域への書き込みはできません。
*/
int Control(const nn::jit::plugin::JitPluginEnvironment& info, uint64_t tag, const void* inData, size_t inDataSize, void* outBuffer, size_t outBufferSize) NN_NOEXCEPT;

/**
    @brief nn::jit::GenerateCode() が呼ばれた際に呼ばれる関数です。

    @param[out] pGeneratedRx 実際に生成された RX コード領域の範囲を格納する変数へのポインタを指定します。
    @param[out] pGeneratedRo 実際に生成された RO コード領域の範囲を格納する変数へのポインタを指定します。
    @param[in] environment Jit プラグイン環境を指定します。
    @param[in] tag タグを指定します。
    @param[in] source コード生成のソースデータを指定します。
    @param[in] sourceSize コード生成のソースデータのサイズを指定します。
    @param[in] rxBuffer RX コードを書き込む領域を指定します。
    @param[in] roBuffer RO コードを書き込む領域を指定します。
    @param[in] inData その他の入力データを指定します。
    @param[in] inDataSize その他の入力データのサイズを指定します。
    @param[in] outBuffer その他の出力を格納するバッファを指定します。
    @param[in] outBufferSize その他の出力を格納するバッファのサイズを指定します。

    @post *pGeneratedRx が rxBuffer に含まれる
    @post *pGeneratedRo が roBuffer に含まれる

    @details
     本関数はアプリケーションで nn::jit::GenrateCode() を呼んだ際に対応して呼ばれます。
*/
int GenerateCode(nn::jit::CodeRange* pGeneratedRx, nn::jit::CodeRange* pGeneratedRo, const nn::jit::plugin::JitPluginEnvironment& environment, uint64_t tag, const void* source, size_t sourceSize, const nn::jit::CodeRange& rxBuffer, const nn::jit::CodeRange& roBuffer, const void* inData, size_t inDataSize, void* outBuffer, size_t outBufferSize) NN_NOEXCEPT;

}}}
