﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <initializer_list>
#include <nn/nn_Common.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <nn/hid/system/hid_InputDetection.h>
#include <nn/omm/omm_Api.h>
#include <nn/applet/applet_FundamentalTypes.h>

namespace nn { namespace idle {

    /**
    * @brief   HandlingContext 構造体内の overrideAutoSleepTime 系パラメータに指定できる値の上界値です。この値自身は指定できません。
    */
    const int MaxOverrideAutoSleepTimeInSeconds = 3600;

    /**
    * @brief   HandlingContext 構造体内の overrideDimmingTime 系パラメータに指定できる値の上界値です。この値自身は指定できません。
    */
    const int MaxOverrideDimmingTimeInSeconds = 900;

    /**
    * @brief   発動延長モード
    */
    enum IdleTimeDetectionExtension : uint32_t
    {
        IdleTimeDetectionExtension_Disabled = 0,
        IdleTimeDetectionExtension_Enabled = 1,
        IdleTimeDetectionExtension_EnabledUnsafe = 2,
    };

    /**
    * @brief   ハンドリングコンテキストを保持する構造体です。
    */
    struct HandlingContext {

        /**
        * @brief   デフォルトコンストラクタです。
        */
        HandlingContext() NN_NOEXCEPT :
            timeMediaPlaybackFinished(),
            overrideAutoSleepTimeInHandheldInSeconds(),
            overrideAutoSleepTimeInConsoleInSeconds(),
            overrideDimmingTimeInHandheldInSeconds(),
            overrideDimmingTimeInConsoleInSeconds(),
            idleTimeDetectionDelayMode(IdleTimeDetectionExtension_Disabled),
            enabledInputSourceIdSet(nn::hid::system::InputSourceIdSet().Set()),
            operationMode(),
            aruid(),
            isAutoSleepDisabled(),
            isDimmingDisabled(),
            isInMediaPlayback()
        {}

        /**
        * @brief   リスト初期化のためのコンストラクタです。
        */
        HandlingContext(
            nn::os::Tick timeMediaPlaybackFinished_,
            int overrideAutoSleepTimeInHandheldInSeconds_,
            int overrideAutoSleepTimeInConsoleInSeconds_,
            int overrideDimmingTimeInHandheldInSeconds_,
            int overrideDimmingTimeInConsoleInSeconds_,
            IdleTimeDetectionExtension idleTimeDetectionDelayMode_,
            nn::hid::system::InputSourceIdSet enabledInputSourceIdSet_,
            nn::omm::OperationMode operationMode_,
            nn::applet::AppletResourceUserId aruid_,
            bool isAutoSleepDisabled_,
            bool isDimmingDisabled_,
            bool isInMediaPlayback_
        ) NN_NOEXCEPT :
            timeMediaPlaybackFinished(timeMediaPlaybackFinished_),
            overrideAutoSleepTimeInHandheldInSeconds(overrideAutoSleepTimeInHandheldInSeconds_),
            overrideAutoSleepTimeInConsoleInSeconds(overrideAutoSleepTimeInConsoleInSeconds_),
            overrideDimmingTimeInHandheldInSeconds(overrideDimmingTimeInHandheldInSeconds_),
            overrideDimmingTimeInConsoleInSeconds(overrideDimmingTimeInConsoleInSeconds_),
            idleTimeDetectionDelayMode(idleTimeDetectionDelayMode_),
            enabledInputSourceIdSet(enabledInputSourceIdSet_),
            operationMode(operationMode_),
            aruid(aruid_),
            isAutoSleepDisabled(isAutoSleepDisabled_),
            isDimmingDisabled(isDimmingDisabled_),
            isInMediaPlayback(isInMediaPlayback_)
        {}

        nn::os::Tick timeMediaPlaybackFinished; //!< フォアグラウンドアプリが最後のメディア再生を終了した時刻を表すシステムチックです。一度も再生していないときは 0 をセットしてください。
        int overrideAutoSleepTimeInHandheldInSeconds; //!< 携帯モード中、フォアグラウンドアプリ上で一時的に自動スリープ発動時間を変更する場合の設定値 [s] です。無効なときは 0 をセットしてください。
        int overrideAutoSleepTimeInConsoleInSeconds; //!< 据置モード中、フォアグラウンドアプリ上で一時的に自動スリープ発動時間を変更する場合の設定値 [s] です。無効なときは 0 をセットしてください。
        int overrideDimmingTimeInHandheldInSeconds; //!< 携帯モード中、フォアグラウンドアプリ上で一時的に自動低輝度化発動時間を変更する場合の設定値 [s] です。無効なときは 0 をセットしてください。
        int overrideDimmingTimeInConsoleInSeconds; //!< 据置モード中、フォアグラウンドアプリ上で一時的に自動低輝度化発動時間を変更する場合の設定値 [s] です。無効なときは 0 をセットしてください。
        IdleTimeDetectionExtension idleTimeDetectionDelayMode; //!< フォアグラウンドアプリ動作中の自動低輝度化・自動スリープ発動遅延が有効になっているかを表します。
        nn::hid::system::InputSourceIdSet enabledInputSourceIdSet; //!< 有効にする入力検知対象を表します。
        nn::omm::OperationMode operationMode; //!< 現在の動作モードです。
        nn::applet::AppletResourceUserId aruid; //!< フォアグラウンドにいるアプリの識別子です。
        bool isAutoSleepDisabled; //!< フォアグラウンドアプリが動作中、自動スリープを無効化するかを表します。
        bool isDimmingDisabled; //!< フォアグラウンドアプリが動作中、自動低輝度化を無効化するかを表します。
        bool isInMediaPlayback; //!< フォアグラウンドアプリが現在メディア再生中であるかを表します。
    };

}}
