﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   I2C デバイスのレジスタへアクセス用のテンプレート関数の定義。
 */

#pragma once

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace i2c {

/**
 * @brief 指定したアドレスのレジスタからデータを読み込みます。
 *
 * @tparam      A            アドレスの型を指定する typename のテンプレート引数。uint8_t, uint16_t, uint32_t, uint64_t をサポート対象とします。
 * @tparam      D            データの型を指定する typename のテンプレート引数。uint8_t, uint16_t, uint32_t, uint64_t をサポート対象とします。
 * @param[out]  pOutData     レジスタから読み込む D 型のデータ
 * @param[in]   session      セッション用パラメータ
 * @param[in]   pAddress     レジスタの A 型のアドレス
 * @return      処理の結果を返します。
 * @retval      ResultSuccess             指定した長さのデータ送信に成功しました。
 * @retval      ResultNoAck               対向デバイスとの接続が確立されていません。デバイスの接続状態を確認ください。
 * @retval      ResultBusBusy             バスが BUSY 状態です。通信間隔が短い場合や同一バスのトランザクションが込み合っている際に発生する可能性があります。
 * @pre         指定したセッションはオープン状態である必要があります。
 * @post        pOutData に対向デバイスのレジスタの値を読み込みます。
 * @details     指定したアドレスのレジスタからデータを読み込むテンプレート関数です。
 *              レジスタのアドレスの型とデータの型はテンプレート引数で指定します。
 */
template <typename A, typename D>
nn::Result ReadSingleRegister(D* pOutData, const nn::i2c::I2cSession& session,
                              const A* pAddress) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pAddress);
    NN_ABORT_UNLESS_NOT_NULL(pOutData);

    const nn::i2c::TransactionOption g_I2cTransStart = nn::i2c::TransactionOption_StartCondition;
    const nn::i2c::TransactionOption g_I2cTransStartStop =
        static_cast<nn::i2c::TransactionOption>(nn::i2c::TransactionOption_StartCondition
                                                | nn::i2c::TransactionOption_StopCondition);

    uint8_t commandList[nn::i2c::CommandListSendCommandSize
        + nn::i2c::CommandListReceiveCommandSize + sizeof(A)];

    nn::i2c::CommandListFormatter commandListFormatter(commandList, sizeof(commandList));

    NN_RESULT_DO(commandListFormatter.EnqueueSendCommand(g_I2cTransStart, pAddress, sizeof(A)));
    NN_RESULT_DO(commandListFormatter.EnqueueReceiveCommand(g_I2cTransStartStop, sizeof(D)));

    NN_RESULT_DO(nn::i2c::ExecuteCommandList(pOutData, sizeof(D), session, commandList,
                                             commandListFormatter.GetCurrentLength()));
    NN_RESULT_SUCCESS;
}

/**
 * @brief 指定したアドレスのレジスタにデータを書き込みます。
 *
 * @tparam      A            アドレスの型を指定する typename のテンプレート引数。uint8_t, uint16_t, uint32_t, uint64_t をサポート対象とします。
 * @tparam      D            データの型を指定する typename のテンプレート引数。uint8_t, uint16_t, uint32_t, uint64_t をサポート対象とします。
 * @param[in]   session      セッション用パラメータ
 * @param[in]   pAddress     レジスタの A 型のアドレス
 * @param[in]   pData        レジスタに書き込む D 型のデータ
 * @return      処理の結果を返します。
 * @retval      ResultSuccess             指定した長さのデータ送信に成功しました。
 * @retval      ResultNoAck               対向デバイスとの接続が確立されていません。デバイスの接続状態を確認ください。
 * @retval      ResultBusBusy             バスが BUSY 状態です。通信間隔が短い場合や同一バスのトランザクションが込み合っている際に発生する可能性があります。
 * @pre         指定したセッションはオープン状態である必要があります。
 * @post        対向デバイスのレジスタに値が書き込まれます。
 * @details     指定したアドレスのレジスタにデータを書き込むテンプレート関数です。
 *              レジスタのアドレスの型とデータの型はテンプレート引数で指定します。
 */
template <typename A, typename D>
nn::Result WriteSingleRegister(const nn::i2c::I2cSession& session,
                               const A* pAddress, const D* pData) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pAddress);
    NN_ABORT_UNLESS_NOT_NULL(pData);

    const nn::i2c::TransactionOption g_I2cTransStartStop =
        static_cast<nn::i2c::TransactionOption>(nn::i2c::TransactionOption_StartCondition
                                                | nn::i2c::TransactionOption_StopCondition);

    uint8_t temporaryBuffer[sizeof(A) + sizeof(D)];

    memcpy(&(temporaryBuffer[0]), pAddress, sizeof(A));
    memcpy(&(temporaryBuffer[sizeof(A)]), pData, sizeof(D));

    NN_RESULT_DO(nn::i2c::Send(session, &(temporaryBuffer[0]), sizeof(A) + sizeof(D), g_I2cTransStartStop));
    NN_RESULT_SUCCESS;
}

}}  // namespace nn::i2c
