﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "i2c_Result.h"
#include "i2c_Type.h"

namespace nn {
namespace i2c {

const size_t CommandListLengthCountMax = 256;  //!< コマンドリストの最大サイズ [byte]
const size_t CommandListReceiveCommandSize = 2; //!< Receive コマンドで消費するコマンドリストのサイズ [byte]
const size_t CommandListSendCommandSize = 2; //!< Send コマンドで消費するコマンドリストのサイズ（送信データ列は含みません） [byte]
const size_t CommandListSleepCommandSize = 2; //!< Sleep コマンドで消費するコマンドリストのサイズ [byte]

/**
 * @brief      コマンドリストを生成するクラスです。@n
 *
 * @details    コンストラクタで受け取ったコマンドリスト作成先のバイト配列に対して、
 *             メンバ関数を通してコマンドを追加します。
 */
class CommandListFormatter
{
    NN_DISALLOW_COPY(CommandListFormatter);
    NN_DISALLOW_MOVE(CommandListFormatter);

public:

    /**
     * @brief   コマンドリスト生成用オブジェクトを構築し初期化します。
     *
     * @param[in]   pCommandList        コマンドリスト格納用配列へのポインタ
     * @param[in]   commandLength       コマンドリスト配列のサイズ[byte]
     *
     * @pre         commandLength は CommandListLengthCountMax 以下である必要があります。
     * @details     コマンドリスト生成用オブジェクトを構築し初期化します。
     *              引数指定されたコマンドリスト格納用配列に対してコマンドが格納されます。
     */
    CommandListFormatter(void* pCommandList, size_t commandLength) NN_NOEXCEPT;


    ~CommandListFormatter() NN_NOEXCEPT
    {
        m_pCommandList = nullptr;
    }

    /**
     * @brief       追加済コマンドリストのサイズを返します。
     *
     * @details     自インスタンスが持つ追加済コマンドリストのサイズ [byte] を返します。
     *
     * @return      追加済コマンドリストのサイズ [byte]
     *
     */
    size_t GetCurrentLength() const { return m_CurrentIndex; }

    /**
     * @brief       コマンドリスト格納用配列の先頭アドレスを返します。
     *
     * @details     自インスタンスが持つコマンドリスト格納用配列の先頭アドレスを返します。
     *
     * @return      コマンドリスト格納配列の先頭アドレス
     *
     */
    const void* GetListHead() const { return m_pCommandList;  }

    /**
     * @brief       受信コマンドをコマンドリストに追加します。
     *
     * @param[in]   inOption               受信時のオプションパラメータ
     * @param[in]   dataBytes              受信するデータ列の長さ [byte]
     * @return      処理の結果を返します。
     * @retval      ResultSuccess          コマンドの追加に成功しました。
     * @retval      ResultCommandListFull  コマンドリストが一杯のためコマンドの追加に失敗しました。
     * @pre         受信サイズは 255 [byte] 以下である必要があります。
     * @details     受信コマンドをコマンドリストに追加します。
     *              コマンドリストは受信するデータ列の長さに関わらず 2 [byte] 消費されます。
     *              受信したデータを格納するバッファは ExecuteCommandList() で指定されます。
     */
    Result EnqueueReceiveCommand( nn::i2c::TransactionOption inOption, size_t dataBytes ) NN_NOEXCEPT;

    /**
     * @brief 送信コマンドをコマンドリストに追加します。
     *
     * @param[in]   inOption               送信時のオプションパラメータ
     * @param[in]   pInData                送信するデータ列
     * @param[in]   dataBytes              送信するデータ列の長さ [byte]
     * @return      処理の結果を返します。
     * @retval      ResultSuccess          コマンドの追加に成功しました。
     * @retval      ResultCommandListFull  コマンドリストが一杯のためコマンドの追加に失敗しました。
     * @pre         送信サイズは 255 [byte] 以下である必要があります。
     * @details     送信コマンドをコマンドリストに追加します。
     *              コマンドリストは 送信データ列の長さ + 2 [byte] 消費されます。
     */
    Result EnqueueSendCommand   ( nn::i2c::TransactionOption inOption, const nn::Bit8* pInData, size_t dataBytes ) NN_NOEXCEPT;

    /**
     * @brief スリープコマンドをコマンドリストに追加します。
     *
     * @param[in]   sleepMicroSeconds      スリープ時間の長さ [us]
     * @return      処理の結果を返します。
     * @retval      ResultSuccess          コマンドの追加に成功しました。
     * @retval      ResultCommandListFull  コマンドリストが一杯のためコマンドの追加に失敗しました。
     * @pre         スリープ時間は 255 [us] 以下である必要があります。
     * @details     スリープコマンドをコマンドリストに追加します。
     *              コマンドリストは 2 [byte] 消費されます。
     */
    Result EnqueueSleepCommand  ( int sleepMicroSeconds ) NN_NOEXCEPT;

private:
    Result IsEnqueueAble( size_t enqueueBytes ) const NN_NOEXCEPT;

private:
    size_t       m_CurrentIndex;        //!< 追加済コマンドリストのサイズを表します。
    size_t       m_CommandListLength;   //!< コマンドリスト配列のサイズ
    void*        m_pCommandList;        //!< コマンドリスト格納用配列へのポインタ
};

} // i2c
} // nn
