﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
* @file
* @brief   I2C ドライバ実装提供者向けのインタフェースクラスの宣言
*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_DeviceCode.h>

#include <nn/ddsf/ddsf_IDriver.h>
#include <nn/i2c/i2c_Type.h>

namespace nn { namespace i2c { namespace driver {

class I2cDeviceProperty;

//! @name I2C ドライバ実装提供者向け API
//! @{

/**
 * @brief   I2C ドライバ実装が備える必要のあるインタフェースクラスです。
 *
 * @detail
 *   I2C ドライバ実装提供者は、本クラスを public 継承したオブジェクトを実装し、 @ref nn::i2c::driver::RegisterDriver() で I2C サブシステムに登録してください。 @n
 *   @n
 *   本クラスから派生クラスへのダウンキャストには、基底クラス @ref nn::ddsf::IDriver の提供する SafeCastTo<>(), SafeCastToPointer<>() メソッドを使用することを強く推奨します。 @n
 *   安全なダウンキャストを使用するには、派生クラスで @ref NN_DDSF_CAST_SAFE_DECL マクロや @ref NN_DDSF_CAST_SAFE_DEFINE マクロを使用した型情報タグの定義が必要です。 @n
 *   詳細は @ref nn::ddsf::ICastSafe のリファレンスを参照してください。
 */
class II2cDriver : public nn::ddsf::IDriver
{
    NN_DDSF_CAST_SAFE_DECL;

    NN_DISALLOW_COPY(II2cDriver);
    NN_DISALLOW_MOVE(II2cDriver);

public:

    //! @name 生成と破棄
    //! @{

    /**
     * @brief   コンストラクタです。
     */
    II2cDriver() NN_NOEXCEPT {}

    /**
     * @brief   デストラクタです。
     */
    virtual ~II2cDriver() NN_NOEXCEPT {}

    //! @}

    //! @name I2C ドライバ実装の初期化・終了
    //! @{

    /**
     * @brief   ドライバを初期化します。
     *
     * @detail
     *   ドライバを初期化します。 @n
     *   このメソッドは @ref nn::i2c::driver::Initialize() の内部で呼ばれます。 @n
     *   @n
     *   このメソッドは、派生クラスでの実装が必須です。
     */
    virtual void InitializeDriver() NN_NOEXCEPT = 0;

    //! @}

    //! @name I2C デバイスの初期化・終了
    //! @{

    /**
     * @brief   I2C デバイスを初期化します。
     *
     * @param[in,out]    pDevice   I2C デバイス
     *
     * @return
     *   処理に成功した場合 @ref nn::ResultSuccess() を、失敗した場合はそれ以外のリザルトコードを返してください。
     *`
     * @pre
     *   - pDevice != nullptr
     *
     * @detail
     *   I2C デバイスを初期化します。 @n
     *   このメソッドは @ref nn::i2c::driver::OpenSession() の内部で、最初にデバイスに対するセッションが開かれたときに呼ばれます。 @n
     *   @n
     *   派生クラスでオーバーライドしない場合、このメソッドは何もせず @ref nn::ResultSuccess() を返します。
     */
    virtual nn::Result InitializeDevice(I2cDeviceProperty* pDevice) NN_NOEXCEPT
    {
        NN_UNUSED(pDevice);
        return nn::ResultSuccess();
    }

    /**
     * @brief   I2C デバイスをファイナライズします。
     *
     * @param[in,out]    pDevice   I2C デバイス
     *
     * @pre
     *   - pDevice != nullptr
     *
     * @detail
     *   I2C デバイスをファイナライズします。 @n
     *   このメソッドは @ref nn::i2c::driver::CloseSession() の内部で、デバイスに対するオープン状態のセッションが一つもなくなったときに呼ばれます。 @n
     *   @n
     *   派生クラスでオーバーライドしない場合、このメソッドは何もしません。
     */
    virtual void FinalizeDevice(I2cDeviceProperty* pDevice) NN_NOEXCEPT
    {
        NN_UNUSED(pDevice);
    }

    //! @}

    //! @name I2C ドライバ実装の送信・受信
    //! @{

    /**
     * @brief 指定したデータ列を送信します。
     *
     * @param[in,out] pDevice I2C デバイス
     * @param[in] pInData 指定したデータ列
     * @param[in] dataBytes 指定したデータ長
     * @param[in] inOption オプション
     *
     * @pre
     *   - pDevice != nullptr
     *   - pInData != nullptr
     *   - dataBytes > 0
     *
     * @return 処理の結果
     *
     * @detail 指定したデータ列を送信します。
     */
    virtual nn::Result Send(I2cDeviceProperty* pDevice, const void* pInData, size_t dataBytes, TransactionOption inOption) NN_NOEXCEPT = 0;

    /**
     * @brief 指定した長さに達するまでデータ列を受信します。
     *
     * @param[out] pOutData 受信バッファ
     * @param[in,out] pDevice I2C デバイス
     * @param[in] dataBytes 指定したデータ長
     * @param[in] inOption オプション
     *
     * @pre
     *   - pOutData != nullptr
     *   - pDevice != nullptr
     *   - dataBytes > 0
     *
     * @return 処理の結果
     *
     * @detail 指定した長さに達するまでデータ列を受信します。
     */
    virtual nn::Result Receive(void* pOutData, I2cDeviceProperty* pDevice, size_t dataBytes, TransactionOption inOption) NN_NOEXCEPT = 0;

    /**
     * @brief 一連の送受信処理を行う間にロックするべきミューテックスを取得します。
     *
     * @return 一連の送受信処理を行う間にロックするべきミューテックスへの参照
     *
     * @detail 指定した長さに達するまでデータ列を受信します。
     */
    virtual nn::os::SdkMutex& GetTransactionOrderMutex() NN_NOEXCEPT = 0;

    //! @}

    //! @name I2C ドライバ実装のサスペンド・レジューム
    //! @{

    /**
     * @brief バスをサスペンドします。
     *
     * @post
     *  バスがサスペンド状態になります。@n
     *  ただし、電源制御に必要なバス（PowerBus）として使用されているバスはサスペンド状態になりません。
     *
     * @details
     *  バスをサスペンドします。@n
     *  ただし、PowerBus として使用されているバスはサスペンドしません。@n
     *  別途、SuspendPowerBus() を呼び出してサスペンド状態にしてください。@n
     *  @n
     *  派生クラスでオーバーライドしない場合、このメソッドは何もしません。
     */
    virtual void SuspendBus() NN_NOEXCEPT {}

    /**
     * @brief 電源制御に必要なバス（PowerBus）をサスペンドします。
     *
     * @post
     *  PowerBus として使用されているバスがサスペンド状態になります。
     *
     * @details
     *  PowerBus として使用されているバスをサスペンドします。@n
     *  @n
     *  派生クラスでオーバーライドしない場合、このメソッドは何もしません。
     */
    virtual void SuspendPowerBus() NN_NOEXCEPT {}

    /**
     * @brief バスをレジュームします。
     *
     * @post
     *  バスがサスペンド前の状態になります。@n
     *  ただし、電源制御に必要なバス（PowerBus）として使用されているバスはサスペンド前の状態になりません。
     *
     * @details
     *  バスをレジュームします。@n
     *  ただし、PowerBus として使用されているバスはレジュームしません。@n
     *  別途、ResumePowerBus() を呼び出してサスペンド前の状態にしてください。@n
     *  @n
     *  派生クラスでオーバーライドしない場合、このメソッドは何もしません。
     */
    virtual void ResumeBus() NN_NOEXCEPT {}

    /**
     * @brief 電源制御に必要なバス（PowerBus）をレジュームします。
     *
     * @post
     *  PowerBus として使用されているバスがサスペンド前の状態になります。
     *
     * @details
     *  PowerBus として使用されているバスをレジュームします。@n
     *  @n
     *  派生クラスでオーバーライドしない場合、このメソッドは何もしません。
     */
    virtual void ResumePowerBus() NN_NOEXCEPT {}

    /**
     * @brief バスのデバイスコードを取得します。
     *
     * @return デバイスコード
     *
     * @details バスのデバイスコードを取得します。
     */
    virtual const nn::DeviceCode& GetDeviceCode() const NN_NOEXCEPT = 0;

    //! @}
};

//! @}

}}} // nn::i2c::driver
