﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>
#include <curl/curl.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Result.h>
#include <nn/ssl/ssl_Context.h>
#include <nn/util/util_Execution.h>

namespace nn { namespace http { namespace stream {

// ヘッダでのエラー発生時の返り値定義
const size_t CallbackErrorValue = static_cast<size_t>(-1);

// context は初期化済みのものが渡ってくる。
typedef Result (*SslContextHandler)(CURL* curlHandle, ssl::Context& context, void* userData);

// HTTP ヘッダからステータスコードを取得するユーティリティ
bool TryExtractStatusCodeFromHeader(long* pOut, const char* header, const size_t Length) NN_NOEXCEPT;

class WebApiAccessorBase
    : public util::CancelInjectable
{
    NN_STATIC_ASSERT(LIBCURL_VERSION_NUM >= 0x072a01); // サポートバージョン

public:
    struct ErrorHandler
    {
        Result(*curlErrorHandler)(CURLcode e);
        Result(*httpStatusHandler)(long httpStatus);
    };

private:
    CURL* m_CurlHandle;
    ErrorHandler m_ErrorHandler {nullptr, nullptr};
    bool m_Initialized {false};

    struct SslCtxFunctionContext
    {
        bool isCalled;
        Result result;
        SslContextHandler handler;
        void* userData;

        static const SslCtxFunctionContext Default;
    } m_SslCtxFunctionContext;

    std::atomic<bool> m_IsStatusCodeExtracted {false};
    long m_StatusCode;

    Result GetSslResultFromHandle(CURLINFO curlInfo) NN_NOEXCEPT;
    Result HandleCurlError(CURLcode e) const NN_NOEXCEPT;
    Result HandleHttpStatus(long status) const NN_NOEXCEPT;

protected:
    static size_t HeaderFunction(char* ptr, size_t unitBytes, size_t count, void* pContext) NN_NOEXCEPT;
    static size_t WriteFunction(char* ptr, size_t unitBytes, size_t count, void* pContext) NN_NOEXCEPT;
    static size_t ReadFunction(char* ptr, size_t unitBytes, size_t count, void* pContext) NN_NOEXCEPT;
    static int XferInfoFunction(void* pContext, curl_off_t dlTotal, curl_off_t dlNow, curl_off_t ulTotal, curl_off_t ulNow) NN_NOEXCEPT;
    static CURLcode DefaultSslCtxFunction(CURL* pCurl, void* pSslContext, SslCtxFunctionContext* pFunctionContext) NN_NOEXCEPT;

    WebApiAccessorBase(CURL* curlHandle, const util::Cancelable* pCancelable) NN_NOEXCEPT;
    ~WebApiAccessorBase() NN_NOEXCEPT;

    Result Attach(CURLM* multi) NN_NOEXCEPT;
    Result Detach(CURLM* multi) NN_NOEXCEPT;
    Result Pause(int bitmask) NN_NOEXCEPT;
    Result Perform() NN_NOEXCEPT;
    Result HandleCurlErrorWithContext(CURLcode e) NN_NOEXCEPT;

    Result GetResult() const NN_NOEXCEPT;
    long GetHttpCode() const NN_NOEXCEPT;

    Result SetDefault() NN_NOEXCEPT;
    void Reset() NN_NOEXCEPT;

    Result SetHeaderFunction(size_t(*writeFunction)(char*, size_t, size_t, void*), void* userData) NN_NOEXCEPT;
    Result SetWriteFunction(size_t(*writeFunction)(char*, size_t, size_t, void*), void* userData) NN_NOEXCEPT;
    Result SetReadFunction(size_t(*readFunction)(char*, size_t, size_t, void*), size_t size, void* userData) NN_NOEXCEPT;
    Result SetSslContextHandler(SslContextHandler handler, void* userData) NN_NOEXCEPT;
    Result SetOpenSocketFunction(curl_socket_t (*openSocketFunction)(void*, curlsocktype, struct curl_sockaddr*), void* userData = nullptr) NN_NOEXCEPT;
    Result SetUrl(const char* url) NN_NOEXCEPT;
    Result SetHeaders(const struct curl_slist* headers) NN_NOEXCEPT;
    Result SetTimeoutSeconds(int32_t seconds) NN_NOEXCEPT;
    Result SetHttpDelete() NN_NOEXCEPT;
    Result SetHttpPatch(const char* patchData, bool copy = false) NN_NOEXCEPT;
    Result SetHttpPost(const char* patchData, bool copy = false) NN_NOEXCEPT;
    Result SetHttpPost(const struct curl_httppost* pFormData) NN_NOEXCEPT;
    Result SetHttpPut(const char* putData, bool copy = false) NN_NOEXCEPT;

    Result SetErrorHandler(ErrorHandler handler) NN_NOEXCEPT;
    Result SetUserAgent(const char* ua) NN_NOEXCEPT;

    Result SetDebugMode() NN_NOEXCEPT;
};

}}}  // ~namespace nn::http::stream
