﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
    @namespace  nn::http::json
    @brief      http ライブラリの JSON 関連機能を提供する名前空間です。
 */

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>

namespace nn { namespace http { namespace json {

// JsonPath 内部で使用する型
struct Node
{
    static const Node Root;
    static const Node Object;
    static const Node Array;
    static const Node Key;

    enum class Kind : int8_t
    {
        Root, Object, Array, Key,
    } kind;
    uint16_t arrayElementCount;

    int16_t Print(char* str, size_t bufferSize, const char* key = nullptr) const NN_NOEXCEPT;
    int16_t GetLength(const char* key = nullptr) const NN_NOEXCEPT
    {
        return Print(nullptr, 0, key);
    }
    void IncrementArrayElementCount() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(kind == Kind::Array);
        arrayElementCount ++;
    }
};

/** @brief JSONパス文字列を表現するクラス

    @tparam NodeDepthMax
        JSONパスとして扱える最大の深さを表します。
        深さは次の要素に対してそれぞれ 1 数えられます。
        - ドキュメントルート, "$"
        - object, "."
        - ペアの string, 任意文字列
        - array, "[*]"
        .
        従って、JSON ドキュメント `{ values : [ 1, 2, 3 ] }` があるとき、 $.values[0] = 1 は深さ 4 です。
        ここで指定した深さを超える場合、次に有効な深さの JSON 文字列が現れるまでパース結果は無視されます。

    @tparam PathLengthMax
        JSONパスとして扱える最大の文字列長さを表します。(終端文字列は含まれません。)
        ここで指定した長さを超える場合、次に有効な長さの JSON 文字列が現れるまでパース結果は無視されます。

    @details
        JSONパス文字列を表現します。
        JSONパス文字列とは、JSON文字列中の任意の値を指すための指定子です。
        例えば次の JSON ドキュメントがあるとき、JSONパス "$.labels[1]" の指す値は "bar" です。

        @code
        {
            labels :
                [
                    "foo", "bar"
                ]
        }
        @endcode

        取り扱うことのできる最大深さと最大長を指定可能であり、これらの制限のいずれかを超える場合、パース結果は無視されます。
        パース結果が無視されていない期間のJSONパス文字列を、ここでは「有効」と表現します。
 */
template <int NodeDepthMax, int PathLengthMax, bool FailOnDepthExcess = true>
class JsonPath
{
    NN_STATIC_ASSERT(NodeDepthMax <= 0xFFFF);
    NN_STATIC_ASSERT(PathLengthMax <= 0xFFFF);

private:
    Node m_Nodes[NodeDepthMax + 1];
    uint16_t m_Heads[NodeDepthMax + 1];
    int32_t m_CurrentDepth;
    int32_t m_InvalidObjectAt;

    char m_Path[PathLengthMax + 1];
    uint16_t m_PathLength;
    bool m_MustCallPop;

    bool IsCurrentNodeValid() const NN_NOEXCEPT;
    bool TryUpdateCurrentNode() NN_NOEXCEPT;
    bool IsPrintable(const Node& node, const char* key = nullptr) const NN_NOEXCEPT;
    bool PopIfKey() NN_NOEXCEPT;
    void UpdatePathValidity() NN_NOEXCEPT;

public:
    JsonPath() NN_NOEXCEPT;

    /** @brief JSONパス文字列を取得します。
        @pre
            - 有効なJSONパス文字列である
     */
    const char* ToString(size_t* pOutLength = nullptr) const NN_NOEXCEPT;
    /** @brief JSONパス文字列が指定した文字列と一致するかを検査します。
        @pre
            - 有効なJSONパス文字列である
     */
    bool Match(const char* path) const NN_NOEXCEPT;
    bool Match(const char* path0, const char* path1) const NN_NOEXCEPT;

    // パーサ内部で使用されるメソッド
    bool TryPush(const Node& node, const char* key = nullptr) NN_NOEXCEPT;
    void Pop() NN_NOEXCEPT;
    bool BeginAcceptValue() NN_NOEXCEPT;
    void EndAcceptValue() NN_NOEXCEPT;
    bool IsValid() NN_NOEXCEPT;
    bool EndObject() NN_NOEXCEPT;
};

}}} // ~namespace nn::http::json

/* --------------------------------------------------------------------------------------------
    実装
 */
#include <cstring>

#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace http { namespace json {

/* --------------------------------------------------------------------------------------------
    JsonPath
*/
#define NN_HTTP_RETURN_FALSE_UNLESS(eval) \
    do \
    { \
        auto _b = (eval); \
        if (!_b) \
        { \
            return false; \
        } \
    } \
    while (NN_STATIC_CONDITION(false))

template <int NodeDepthMax, int PathLengthMax, bool FailOnDepthExcess>
inline JsonPath<NodeDepthMax, PathLengthMax, FailOnDepthExcess>::JsonPath() NN_NOEXCEPT
    : m_CurrentDepth(0)
    , m_InvalidObjectAt(-1)
    , m_PathLength(0)
    , m_MustCallPop(false)
{
    auto root = Node::Root;

    // Root ノードの生成
    NN_SDK_ASSERT(m_CurrentDepth == 0);
    m_Nodes[m_CurrentDepth] = root;
    m_Heads[m_CurrentDepth] = 0;

    // Root パスの生成
    m_PathLength = root.Print(m_Path, sizeof(m_Path));
    NN_SDK_ASSERT(m_PathLength <= PathLengthMax);
}
template <int NodeDepthMax, int PathLengthMax, bool FailOnDepthExcess>
inline const char* JsonPath<NodeDepthMax, PathLengthMax, FailOnDepthExcess>::ToString(size_t* pOutLength) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsCurrentNodeValid()); // 現在のノードは妥当か
    if (pOutLength != nullptr)
    {
        *pOutLength = m_PathLength;
    }
    return m_Path;
}
template <int NodeDepthMax, int PathLengthMax, bool FailOnDepthExcess>
inline bool JsonPath<NodeDepthMax, PathLengthMax, FailOnDepthExcess>::Match(const char* path) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(path);
    NN_SDK_ASSERT(IsCurrentNodeValid()); // 現在のノードは妥当か
    return std::strncmp(m_Path, path, m_PathLength + 1) == 0; // 終端ぶん +1
}
template <int NodeDepthMax, int PathLengthMax, bool FailOnDepthExcess>
inline bool JsonPath<NodeDepthMax, PathLengthMax, FailOnDepthExcess>::Match(const char* path0, const char* path1) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(path0);
    NN_SDK_ASSERT_NOT_NULL(path1);
    NN_SDK_ASSERT(IsCurrentNodeValid()); // 現在のノードは妥当か
    auto length0 = strnlen(path0, m_PathLength);
    return true
        && std::strncmp(m_Path, path0, length0) == 0
        && std::strncmp(m_Path + length0, path1, m_PathLength + 1 - length0) == 0;
}
template <int NodeDepthMax, int PathLengthMax, bool FailOnDepthExcess>
inline bool JsonPath<NodeDepthMax, PathLengthMax, FailOnDepthExcess>::IsCurrentNodeValid() const NN_NOEXCEPT
{
    NN_HTTP_RETURN_FALSE_UNLESS(m_CurrentDepth <= NodeDepthMax);
    NN_HTTP_RETURN_FALSE_UNLESS(m_InvalidObjectAt < 0); // 無効なオブジェクトではない
    return true;
}
template <int NodeDepthMax, int PathLengthMax, bool FailOnDepthExcess>
inline bool JsonPath<NodeDepthMax, PathLengthMax, FailOnDepthExcess>::TryUpdateCurrentNode() NN_NOEXCEPT
{
    auto& node = m_Nodes[m_CurrentDepth];
    if (node.kind == Node::Kind::Array)
    {
        auto head = m_Heads[m_CurrentDepth];
        m_PathLength = head + node.Print(m_Path + head, sizeof(m_Path) - head);
        node.IncrementArrayElementCount(); // Array なら子要素をひとつ増やす
        return m_PathLength <= PathLengthMax;
    }
    return true;
}
template <int NodeDepthMax, int PathLengthMax, bool FailOnDepthExcess>
inline bool JsonPath<NodeDepthMax, PathLengthMax, FailOnDepthExcess>::IsPrintable(const Node& node, const char* key) const NN_NOEXCEPT
{
    NN_HTTP_RETURN_FALSE_UNLESS(m_CurrentDepth + 1 <= NodeDepthMax);
    NN_HTTP_RETURN_FALSE_UNLESS(static_cast<int32_t>(m_PathLength) + node.GetLength(key) <= PathLengthMax); // 自身のパスを保持できるか？
    return true;
}
template <int NodeDepthMax, int PathLengthMax, bool FailOnDepthExcess>
inline bool JsonPath<NodeDepthMax, PathLengthMax, FailOnDepthExcess>::PopIfKey() NN_NOEXCEPT
{
    NN_HTTP_RETURN_FALSE_UNLESS(IsCurrentNodeValid());
    NN_HTTP_RETURN_FALSE_UNLESS(m_Nodes[m_CurrentDepth].kind == Node::Kind::Key);
    Pop();
    return true;
}
template <int NodeDepthMax, int PathLengthMax, bool FailOnDepthExcess>
inline bool JsonPath<NodeDepthMax, PathLengthMax, FailOnDepthExcess>::TryPush(const Node& node, const char* key) NN_NOEXCEPT
{
    NN_SDK_ASSERT(node.kind != Node::Kind::Root);
    NN_SDK_ASSERT(!m_MustCallPop);

    // 深さの検査
    NN_HTTP_RETURN_FALSE_UNLESS(!FailOnDepthExcess || m_CurrentDepth + 1 <= NodeDepthMax);

    if (!(IsCurrentNodeValid()      // 現在のノードは有効か
        && TryUpdateCurrentNode()   // 現在のノードを更新して妥当なパス文字列を生成可能か
        && IsPrintable(node, key))) // 新しいノード込みでパス文字列を生成可能か
    {
        // パス文字列の都合上 Push できない場合、
        // Key であれば現在の深さ、Object や Array であれば自身を抜けるまでの間、処理を無効とする。
        if (false
            || node.kind == Node::Kind::Object
            || node.kind == Node::Kind::Array)
        {
            ++ m_CurrentDepth;
        }

        if (m_InvalidObjectAt < 0)
        {
            m_InvalidObjectAt = m_CurrentDepth;
        }
        return true; // 処理は継続する
    }

    // Push 処理
    ++ m_CurrentDepth;
    m_Nodes[m_CurrentDepth] = node;
    auto head = (m_Heads[m_CurrentDepth] = m_PathLength);
    m_PathLength = head + node.Print(m_Path + head, sizeof(m_Path) - head, key); // IsPushable で成功することは保証される。
    NN_SDK_ASSERT(m_PathLength <= PathLengthMax);
    return true;
}
template <int NodeDepthMax, int PathLengthMax, bool FailOnDepthExcess>
inline void JsonPath<NodeDepthMax, PathLengthMax, FailOnDepthExcess>::Pop() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_CurrentDepth > 0);

    if (!m_MustCallPop)
    {
        if (IsCurrentNodeValid())
        {
            NN_SDK_ASSERT(m_Heads[m_CurrentDepth] <= PathLengthMax);
            // パス文字列の更新
            m_PathLength = m_Heads[m_CurrentDepth];
            m_Path[m_PathLength] = '\0';
        }
        else
        {
            UpdatePathValidity();
        }
    }
    m_MustCallPop = false;

    m_CurrentDepth--;
    auto r = PopIfKey();
    NN_UNUSED(r);
}
template <int NodeDepthMax, int PathLengthMax, bool FailOnDepthExcess>
inline bool JsonPath<NodeDepthMax, PathLengthMax, FailOnDepthExcess>::BeginAcceptValue() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_MustCallPop);

    NN_HTTP_RETURN_FALSE_UNLESS(IsCurrentNodeValid()); // 現在のノードは妥当か
    NN_HTTP_RETURN_FALSE_UNLESS(TryUpdateCurrentNode()); // 現在のノードを更新可能か
    NN_SDK_ASSERT(false
        || m_Nodes[m_CurrentDepth].kind == Node::Kind::Array
        || m_Nodes[m_CurrentDepth].kind == Node::Kind::Key,
        "%s (depth=%d)\n", m_Path, m_CurrentDepth);
    return true;
}
template <int NodeDepthMax, int PathLengthMax, bool FailOnDepthExcess>
inline void JsonPath<NodeDepthMax, PathLengthMax, FailOnDepthExcess>::EndAcceptValue() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_MustCallPop);

    auto r = PopIfKey();
    NN_UNUSED(r);
}
template <int NodeDepthMax, int PathLengthMax, bool FailOnDepthExcess>
inline bool JsonPath<NodeDepthMax, PathLengthMax, FailOnDepthExcess>::IsValid() NN_NOEXCEPT
{
    return IsCurrentNodeValid();
}
template <int NodeDepthMax, int PathLengthMax, bool FailOnDepthExcess>
inline void JsonPath<NodeDepthMax, PathLengthMax, FailOnDepthExcess>::UpdatePathValidity() NN_NOEXCEPT
{
    if (m_CurrentDepth <= m_InvalidObjectAt)
    {
        // Invalid 判定された Depth の pop
        m_InvalidObjectAt = -1;
    }
}
template <int NodeDepthMax, int PathLengthMax, bool FailOnDepthExcess>
inline bool JsonPath<NodeDepthMax, PathLengthMax, FailOnDepthExcess>::EndObject() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_CurrentDepth > 0);
    NN_SDK_ASSERT(!m_MustCallPop);

    if (IsCurrentNodeValid())
    {
        return true;
    }

    m_MustCallPop = true;
    UpdatePathValidity();
    return IsCurrentNodeValid();
}
#undef NN_HTTP_RETURN_FALSE_UNLESS
}}} // ~namespace nn::http::json
