﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <nn/http/http_Types.h>

#include <curl/curl.h>

namespace nn
{
namespace http
{

enum MethodType
{
    MethodType_Get,
    MethodType_Head,
    MethodType_Post,
    MethodType_Put,
    MethodType_Delete,
    MethodType_Patch,
    MethodType_Custom,
    MethodType_Max
};

class ConnectionBroker;
class ResponseImpl;

/**
 * @brief   HTTP リクエストを表すクラスです。

 * @details
 *  HTTP リクエストのパラメータを保持します。
 */
class Request
{
    NN_DISALLOW_COPY(Request);
    NN_DISALLOW_MOVE(Request);
public:
    static const size_t FormatBufferSizeDefault = 4096;

    NN_IMPLICIT Request(ConnectionBroker& broker);
    ~Request();

    void SetFormatBufferSize(size_t size);
    void SetTimeout(const TimeSpan& timeout);
    void SetConnectTimeout(const TimeSpan& timeout);
    void SetKeepAlive(bool bEnable);
    void SetVerbose(bool bVerbose);
    void SetSkipSslVerificationForDebug(bool bSkip);

    Result AddHeaderLine(const char * pLine);
    Result AddHeaderFormat(const char* format, ...);
    Result SetCookieFormat(const char* format, ...);
    void SetMethodType(MethodType type);
    const char* GetUrl() const;
    void SetUrlPointer(const char* pUrl);
    Result SetUrlByFormatString(const char* format, ...);
    void SetPostDataPointer(const char* pData, size_t size);
    Result SetPostDataByFormatString(const char* format, ...);

    Result WriteBody(const char* pBody, size_t size);
    Result GetResponse(ResponseImpl* pResponse);

    void SetCancelEvent(os::Event* pCancelEvent);

    static const long RedirectCountMax      = 5L; // リダイレクトは 5 回まで
    static const long LowSpeedLimit         = 1L; // 転送ビットレートは 1bps を下限とする
    static const long LowSpeedTimeSeconds   = 30; // LowSpeedLimit が 30 秒つづくと通信を中断する

    static const long ConnectTimeoutSeconds = 30;  // サーバとの接続(CONNECT)のタイムアウト
    static const long TimeoutSeconds        = 300; // 通信全体でのタイムアウト

protected:
    Result AllocateAndFormatString(char ** ppOut, size_t* pOutSize, const char* format, std::va_list vlist);

private:
    Result ApplyOptionsToEasyHandle(CURL* pCurlEasyHandle);

    Result EnsureEasyHandleReady();

    os::Mutex m_mutexReadWrite;
    const char* m_pUrl;
    char* m_pUrlBuffer;

    nn::TimeSpan m_timeTimeout;
    nn::TimeSpan m_timeConnectTimeout;

    MethodType m_MethodType;
    ConnectionBroker& m_Connection;
    curl_slist* m_pHeaderList;
    CURL* m_pEasyHandle;

    bool m_bKeepAlive;
    bool m_bSkipSslVerificationForDebug;
    bool m_bVerbose;


    size_t m_sizeFormatBuffer;
    //TODO
    const char* m_pPostData;
    size_t m_sizePostData;

    char* m_pPostDataBuffer;
    char* m_pCookieBuffer;
    os::Event* m_pCancelEvent;
};

inline const char* Request::GetUrl() const
{
    return m_pUrl;
}

}
}
