﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
* @file
* @brief   ホスト・ターゲット間ソケット通信のプライベート関数
*/
#include <nn/nn_Allocator.h>
#include <nn/htcs/htcs_Types.h>

namespace nn { namespace htcs {

/**
*   @brief      同時に操作可能なセッション数を指定して、ホスト・ターゲット間ソケット通信を初期化します。@n
*               nn::htcs::GetworkingMemorySize(nn::htcs::SocketCountMax) で返るサイズのメモリが、この関数呼び出し時に alloc を使用して確保されます。@n
*               これ以降はメモリのアロケートは行われません。
*
*   @param[in]  alloc           メモリ確保用の関数を指定します。
*   @param[in]  dealloc         メモリ解放用の関数を指定します。
*   @param[in]  sessionCount    同時に操作可能なセッション数を指定します。
*
*   @pre
*               - sessionCount > 0
*               - sessionCount <= nn::htcs::SessionCountMax
*/
void Initialize(AllocateFunction alloc, DeallocateFunction dealloc, int sessionCount) NN_NOEXCEPT;

/**
*   @brief      同時に操作可能なセッション数を指定して、ホスト・ターゲット間ソケット通信を初期化します。@n
*               こちらで初期化した場合 htcs の切断エミュレーション中も API が使用できるようになります。@n
*               nn::htcs::GetworkingMemorySize(nn::htcs::SocketCountMax) で返るサイズのメモリが、この関数呼び出し時に alloc を使用して確保されます。@n
*               これ以降はメモリのアロケートは行われません。
*
*   @param[in]  alloc           メモリ確保用の関数を指定します。
*   @param[in]  dealloc         メモリ解放用の関数を指定します。
*   @param[in]  sessionCount    同時に操作可能なセッション数を指定します。
*
*   @pre
*               - sessionCount > 0
*               - sessionCount <= nn::htcs::SessionCountMax
*/
void InitializeForDisableDisconnectionEmulation(AllocateFunction alloc, DeallocateFunction dealloc, int sessionCount) NN_NOEXCEPT;

/**
*   @brief      同時に操作可能なセッション数を指定して、ホスト・ターゲット間ソケット通信を初期化します。@n
*               必要なバッファのサイズは nn::htcs::GetWorkingMemorySize で算出できます。
*
*   @param[in]  buffer      割り当てるバッファを指定します。
*   @param[in]  bufferSize  バッファのサイズを指定します。
*   @param[in]  sessionCount    同時に操作可能なセッション数を指定します。
*/
void InitializeForSystem(void* buffer, size_t bufferSize, int sessionCount) NN_NOEXCEPT;

/**
*   @brief      同時に操作可能なセッション数を指定して、ホスト・ターゲット間ソケット通信を初期化します。@n
*               こちらで初期化した場合 htcs の切断エミュレーション中も API が使用できるようになります。
*               必要なバッファのサイズは nn::htcs::GetWorkingMemorySize で算出できます。
*
*   @param[in]  buffer      割り当てるバッファを指定します。
*   @param[in]  bufferSize  バッファのサイズを指定します。
*   @param[in]  sessionCount    同時に操作可能なセッション数を指定します。
*
*   @pre
*               - sessionCount > 0
*               - sessionCount <= nn::htcs::SessionCountMax
*/
void InitializeForDisableDisconnectionEmulation(void* buffer, size_t bufferSize, int sessionCount) NN_NOEXCEPT;
}}

