﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief HDCP ライブラリの API 宣言。
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/hdcp/hdcp_Result.h>
#include <nn/hdcp/hdcp_Type.h>

namespace nn {
namespace hdcp {

    /**
     * @brief HDCPの認証状態の遷移時にシグナルされるイベントを取得します。
     *
     * @param[in] pEvent      状態遷移時にシグナルされるイベントへのポインタ。
     *
     * @return                       実行結果を返します。
     * @retval ResultSuccess         処理に成功しました。
     * @retval ResultOperationFailed 内部エラーで処理に失敗しました。
     * @retval ResultOutOfMemory     メモリ不足で処理に失敗しました。
     *
     * @pre
     *      - HDCP ライブラリが初期化されている必要があります。
     *      - pEvent が指すイベントが初期化されている必要があります。
     *      - @a pEvent != nullptr
     * @post @a pEvent が現在のHDCPの認証状態の遷移時にシグナルされるイベントとなります。
     *
     * @details HDCPの認証状態の遷移時にシグナルされるイベントを取得します。
     */
    Result GetHdcpStateTransitionEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT;

    /**
     * @brief 現在のHDCPの認証状態を取得します。
     *
     * @param[out] pOutState    HDCPの認証状態
     *
     * @return                       実行結果を返します。
     * @retval ResultSuccess         処理に成功しました。
     * @retval ResultOperationFailed 処理に失敗しました。
     *
     * @pre
     *      - HDCP ライブラリが初期化されている必要があります。
     *      - @a pOutState != nullptr
     * @post 現在のHDCPの状態が @a pOutState に格納されます。
     *
     * @details 現在のHDCPの認証状態を取得します。
     */
    Result GetHdcpAuthenticationState(HdcpAuthenticationState* pOutState) NN_NOEXCEPT;

    /**
     * @brief 現在のHDCPの有効/無効の設定を切り替えます。
     *
     * @param[in]  mode         HDCPの有効/無効の設定
     *
     * @return                       実行結果を返します。
     * @retval ResultSuccess         処理に成功しました。
     * @retval ResultOperationFailed 処理に失敗しました。
     *
     * @pre  HDCP ライブラリが初期化されている必要があります。
     * @post @a mode が現在のHDCPの設定となります。
     *
     * @details 現在のHDCPの有効/無効の設定を切り替えます。
     */
    Result SetCurrentHdcpMode(HdcpMode mode) NN_NOEXCEPT;

    /**
     * @brief 現在のHDCPの有効/無効の設定を取得します。
     *
     * @pre HDCP ライブラリが初期化されている必要があります。
     *
     * @details 現在のHDCPの有効/無効の設定を取得します。
     */
    HdcpMode GetCurrentHdcpMode() NN_NOEXCEPT;

} // hdcp
} // nn
