﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_DeviceCode.h>
#include <nn/nn_SdkAssert.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/dt.h>

#include <nn/gpio/detail/gpio_Log.h>

namespace nn { namespace gpio { namespace driver { namespace detail { namespace dt {

    /**
    * @brief デバイスコードノードパーサのインタフェースクラスです。
    *
    * @detail
    *   デバイスコードノードは DeviceTree 上でデバイスコードとハードウェア上の実体の対応を表現するノードです。 @n
    *   各ノードは 1 つ以上のデバイスコードのリストと compatible プロパティを持ち、対応するドライバに固有のパラメータを保持することができます。 @n
    *   ドライバはこのクラスを継承して実装したパーサオブジェクトを、ルートパーサ ( @ref DeviceCodeNodeRootParser ) に登録します。 @n
    *   ルートパーサが DeviceTree をパースする際、自分の対応する compatible 名をプロパティに持つデバイスコードノードを発見すると、@n
    *   ノード内のデバイスコードごとに所定のメソッドが呼び出されます。 @n
    *   @n
    *   ルートパーサによるパース ( @ref DeviceCodeNodeRootParser::ParseAll() ) の中では、以下のような流れで本クラスの実装必須メソッド群が呼び出されます。 @n
    *
    *   1. 一つのデバイスコードノードを読み込み、compatible プロパティと本クラスの構築時に指定した compatible 名の文字列を比較します。
    *   2. 一致する compatible 名を持つデバイスコードノードが見つかった場合、 OnCompatibleNodeBegin() が呼び出されます。
    *       * このメソッドの中で、各ノードのパース開始時の準備処理を実装してください。
    *       * このメソッドが失敗を返した場合、このノードのパースは行われず、次のノードにスキップします。このとき OnCompatibleNodeEnd() は呼ばれません。
    *       * このメソッドが成功を返した場合、次のステップに移行します。
    *   3. このノードのデバイスコードのリストが格納されている device-codes プロパティが先頭から一つずつ読まれ、デバイスコードごとに OnDeviceCodeFound() が呼び出されます。
    *       * このメソッドが失敗を返した場合、このノードのパースはここで終了します。最後に OnCompatibleNodeEnd() が呼ばれます。
    *       * ノード内のすべてのデバイスコードについてメソッドを呼び出した後、次のステップに移行します。
    *   4. 最後にこのノードについて OnCompatibleNodeEnd() が呼び出されます。
    *   5. 次のデバイスコードノードがあれば、 1 に戻ります。
    */
    class IDeviceCodeNodeParser
    {
        NN_DISALLOW_COPY(IDeviceCodeNodeParser);
        NN_DISALLOW_MOVE(IDeviceCodeNodeParser);

    private:
        nn::util::IntrusiveListNode m_ListNode;
        friend class nn::util::IntrusiveListMemberNodeTraits<IDeviceCodeNodeParser, &IDeviceCodeNodeParser::m_ListNode>;

    public:
        typedef nn::util::IntrusiveList<
            IDeviceCodeNodeParser, nn::util::IntrusiveListMemberNodeTraits<IDeviceCodeNodeParser, &IDeviceCodeNodeParser::m_ListNode>
        > List;

    public:
        /**
        * @brief    コンストラクタです。
        *
        * @tparam   N     compatible 名の配列の要素数です。通常直接指定する必要はありません。
        *
        * @param[in]    compatibleNames  対応する compatible 名の配列
        *
        * @pre
        *   - compatibleNames != nullptr
        *   - compatibleNames の要素数が 1 以上
        *
        * @detail
        *   コンストラクタです。 @n
        *   派生クラスのコンストラクタ内で、以下のコード例のように対応する compatible 名の配列を渡して初期化してください。
        *
        * @code
        *   class DeviceCodeNodeParserImpl : public IDeviceCodeNodeParser
        *   {
        *   public:
        *       DeviceCodeNodeParserImpl() NN_NOEXCEPT :
        *          IDeviceCodeNodeParser({ "nintendo,specific-chip001" })
        *       {}
        *
        *       // ...
        *   }
        * @endcode
        */
        template<size_t N>
        explicit IDeviceCodeNodeParser(const char* const (&compatibleNames)[N]) NN_NOEXCEPT :
            m_ppCompatibleNames(&compatibleNames[0]),
            m_CompatibleNameCount(N)
        {
            NN_STATIC_ASSERT(N > 0);
        }

        /**
        * @brief    デストラクタです。
        */
        virtual ~IDeviceCodeNodeParser() NN_NOEXCEPT {}

        /**
        * @brief    渡された文字列が、本クラスのサポートする compatible 名に一致するかを判定します。
        *
        * @param[in]    valueInProperty  ノードプロパティに格納されていた compatible 名
        *
        * @return   本クラスが保持する compatible 名の中に @a valueInProperty と完全一致するものがあれば true を返します。
        *
        * @pre
        *   - valueInProperty != nullptr
        *
        * @detail
        *   渡された文字列が、本クラスのサポートする compatible 名に一致するかを判定します。
        */
        bool IsCompatible(const char* valueInProperty) const NN_NOEXCEPT;

        /**
        * @brief    本クラスが対応するデバイスコードノードのパース開始時に呼ばれるメソッドです。
        *
        * @param[in]    node  デバイスコードノード
        *
        * @return
        *   パース開始準備の処理に成功した場合 nn::ResultSuccess() を、失敗した場合はそれ以外のリザルトコードを返してください。
        *
        * @detail
        *   本クラスが対応する（compatible 名が一致する）デバイスコードノードが見つかった際に、パースの開始前に呼び出されます。 @n
        *   このメソッドが成功を返した場合、続いてノードのパースが行われます。 @n
        *   このメソッドが失敗を返した場合、このノードのパースは行われず、次のノードにスキップします。このとき @a OnCompatibleNodeEnd() も呼ばれません。 @n
        *   派生クラスでオーバーライドしない場合、このメソッドは何もせず nn::ResultSuccess() を返します。
        */
        virtual nn::Result OnCompatibleNodeBegin(nn::dt::Node node) NN_NOEXCEPT
        {
            NN_UNUSED(node);
            NN_RESULT_SUCCESS;
        }

        /**
        * @brief    本クラスが対応するデバイスコードノードのパース終了時に呼ばれるメソッドです。
        *
        * @param[in]    node  デバイスコードノード
        *
        * @detail
        *   本クラスが対応するデバイスコードノードのパース終了時に呼ばれるメソッドです。 @n
        *   派生クラスでオーバーライドしない場合、このメソッドは何もせず nn::ResultSuccess() を返します。
        */
        virtual void OnCompatibleNodeEnd(nn::dt::Node node) NN_NOEXCEPT
        {
            NN_UNUSED(node);
        }

        /**
        * @brief    デバイスコードノード中に見つかったデバイスコードを通知するメソッドです。
        *
        * @param[in]    deviceCode  デバイスコード
        * @param[in]    node        デバイスコードノード
        *
        * @return
        *   通知されたデバイスコードに関する必要な処理に成功した場合 nn::ResultSuccess() を、失敗した場合はそれ以外のリザルトコードを返してください。
        *
        * @detail
        *   このメソッドは、本クラスが @e OnCompatibleNodeBegin() で成功を返したノードについて、device-codes プロパティに記述されているデバイスコードごとに呼び出されます。 @n
        *   呼び出しは device-codes プロパティの先頭から順に行われます。 @n
        *   このメソッドが失敗を返した場合、このノードについてのパースはその時点で終了します。このときも、最後に @a OnCompatibleNodeEnd() が呼ばれます。 @n
        *   このメソッドは、派生クラスでの実装が必須です。
        */
        virtual nn::Result OnDeviceCodeFound(nn::DeviceCode deviceCode, nn::dt::Node node) NN_NOEXCEPT = 0;

    private:
        const char* const* m_ppCompatibleNames;
        const int m_CompatibleNameCount;
    };

    /**
    * @brief デバイスコードノードのルートパーサです。
    *
    * @detail
    *   デバイスコードノードは DeviceTree 上でデバイスコードとハードウェア上の実体の対応を表現するノードです。 @n
    *   本クラスは、DeviceTree 上の所定のノード下に含まれるデバイスコードノードを探索し、 @n
    *   条件に合致したノードごとに、本クラスに登録された個別のノードパーサのメソッドを呼び出す機能を提供します。
    */
    class DeviceCodeNodeRootParser
    {
        NN_DISALLOW_COPY(DeviceCodeNodeRootParser);
        NN_DISALLOW_MOVE(DeviceCodeNodeRootParser);

    public:
        //! サブシステムタグのプロパティ名 ("subsys-<subsystem>") を格納するバッファサイズです。
        static constexpr int MaxSubsystemTagNameBytes = 16;

        //! compatible プロパティの最大長です。
        static constexpr int MaxCompatibleBytes = 128;

    public:
        /**
        * @brief    コンストラクタです。
        *
        * @param[in]    subsystemTag  対応するサブシステムタグのプロパティ名 ("subsys-<subsystem>")
        *
        * @pre
        *   - subsystemTag != nullptr
        *   - strlen(subsystemTag) < MaxSubsystemTagNameBytes
        *
        * @detail
        *   コンストラクタです。
        */
        explicit DeviceCodeNodeRootParser(const char* subsystemTag) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_LESS(strnlen(subsystemTag, NN_ARRAY_SIZE(m_SubsystemTag)), NN_ARRAY_SIZE(m_SubsystemTag));
            strncpy(m_SubsystemTag, subsystemTag, NN_ARRAY_SIZE(m_SubsystemTag));

            m_ParserList.clear();
        }

        /**
        * @brief    デバイスコードノードパーサを登録します。
        *
        * @param[in]    pParser  デバイスコードノードパーサを登録します。
        *
        * @pre
        *   - pParser != nullptr
        *
        * @detail
        *   デバイスコードノードパーサを登録します。
        */
        void RegisterParser(IDeviceCodeNodeParser* pParser) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pParser);

            std::lock_guard<decltype(m_ParserListLock)> lock(m_ParserListLock);
            m_ParserList.push_back(*pParser);
        }

        /**
        * @brief    デバイスコードノードパーサの登録を解除します。
        *
        * @param[in]    pParser  デバイスコードノードパーサを登録します。
        *
        * @pre
        *   - pParser != nullptr
        *
        * @detail
        *   デバイスコードノードパーサの登録を解除します。
        */
        void UnregisterParser(IDeviceCodeNodeParser* pParser) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pParser);

            std::lock_guard<decltype(m_ParserListLock)> lock(m_ParserListLock);
            m_ParserList.erase(m_ParserList.iterator_to(*pParser));
        }

        /**
        * @brief    デバイスコードノードのパースを行います。
        *
        * @return
        *   パースが完了した場合 nn::ResultSuccess() を、パース中に予期せぬエラーが発生し、パースを完了できなかった場合にそれ以外を返します。
        *
        * @detail
        *   デバイスコードノードのパースを行います。 @n
        *   内部で起こるパースの詳細な流れについては @ref IDeviceCodeNodeParser のリファレンスを参照してください。
        */
        nn::Result ParseAll() NN_NOEXCEPT;

    private:
        // ノードがパース対象であるかをチェック
        nn::Result CheckSubsystemTag(bool* pOutIsTagMatched, nn::dt::Node node) NN_NOEXCEPT;

        // ノードの compatible プロパティを読み、登録済パーサから対応するものを見つける
        nn::Result FindCompatibleParser(IDeviceCodeNodeParser** ppOutParser, nn::dt::Node node) NN_NOEXCEPT;

        // デバイスコードノードを一つパースする
        // 一つのノードに複数のデバイスコードが含まれる
        nn::Result ParseSingleNode(nn::dt::Node node) NN_NOEXCEPT;

    private:
        char m_SubsystemTag[MaxSubsystemTagNameBytes];
        IDeviceCodeNodeParser::List m_ParserList;
        nn::os::SdkMutex m_ParserListLock;
    };

}}}}} // nn::gpio::driver::detail::dt
