﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
*   @file
*   @brief  セーブデータ関連 API の宣言
*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/account/account_Types.h>

namespace nn { namespace fs {

//! @name セーブデータ関連 API
//! @{

//--------------------------------------------------------------------------
/**
*   @brief      セーブデータサムネイルファイルからデータを読み込みます。
*
*   @param[in]  saveDataOwnerId セーブデータの所有者を示す ID
*   @param[in]  user            対象のユーザーを指す Uid
*   @param[in]  headerData      読み込みヘッダデータの格納先バッファ
*   @param[in]  headerSize      読み込むヘッダデータのバイトサイズ
*   @param[in]  bodyData        読み込みボディデータの格納先バッファ
*   @param[in]  bodySize        読み込むボディデータのバイトサイズ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*   @retval     ResultTargetNotFound        サムネイルファイルと関連付けられるセーブデータが作成されていません。
*   @retval     ResultDataCorrupted     サムネイルファイルが壊れています。
*
*   @pre
*       - headerData, bodyData が有効なメモリを指している。
*       - headerSize が headerData のサイズ以下である。
*       - bodySize が bodyData のサイズ以下である。
*       - nn::account::Initialize() でアカウントシステムのライブラリが初期化済みである。
*
*   @details    ユーザーセーブデータに関連付くサムネイルファイルからデータを読み込みます。@n
*               headerSize, bodySize が対象のサムネイルファイルに対する直前の nn::fs::WriteSaveDataThumbnailFile() または
*               nn::fs::WriteSaveDataThumbnailFileHeader() 成功時の値と異なる場合、ResultDataCorrupted が返り正しいデータが読み込めません。@n
*               ヘッダデータが未書き込みのサムネイルファイルに対して実行した場合、headerData が 0 埋めされて処理が成功します。@n
*               ボディデータが未書き込みのサムネイルファイルに対して実行した場合、bodyData が 0 埋めされて処理が成功します。
*/
Result ReadSaveDataThumbnailFile(Bit64 saveDataOwnerId, const nn::account::Uid& user, void* headerData, size_t headerSize, void* bodyData, size_t bodySize) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      セーブデータサムネイルファイルからヘッダデータを読み込みます。
*
*   @param[in]  saveDataOwnerId セーブデータの所有者を示す ID
*   @param[in]  user            対象のユーザーを指す Uid
*   @param[in]  headerData      読み込みヘッダデータの格納先バッファ
*   @param[in]  headerSize      読み込むヘッダデータのバイトサイズ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*   @retval     ResultTargetNotFound        サムネイルファイルと関連付けられるセーブデータが作成されていません。
*   @retval     ResultDataCorrupted     サムネイルファイルが壊れています。
*
*   @pre
*       - headerData が有効なメモリを指している。
*       - headerSize が headerData のサイズ以下である。
*       - nn::account::Initialize() でアカウントシステムのライブラリが初期化済みである。
*
*   @details    ユーザーセーブデータに関連付くサムネイルファイルからヘッダデータを読み込みます。@n
*               headerSize が対象のサムネイルファイルに対する直前の nn::fs::WriteSaveDataThumbnailFile() または
*               nn::fs::WriteSaveDataThumbnailFileHeader() 成功時の値と異なる場合、ResultDataCorrupted が返り正しいデータが読み込めません。@n
*               ヘッダデータが未書き込みのサムネイルファイルに対して実行した場合、headerData が 0 埋めされて処理が成功します。
*/
Result ReadSaveDataThumbnailFileHeader(Bit64 saveDataOwnerId, const nn::account::Uid& user, void* headerData, size_t headerSize) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      セーブデータサムネイルファイルにデータを書き込みます。
*
*   @param[in]  saveDataOwnerId セーブデータの所有者を示す ID
*   @param[in]  user            対象のユーザーを指す Uid
*   @param[in]  headerData      読み込みヘッダデータの格納先バッファ
*   @param[in]  headerSize      読み込むヘッダデータのバイトサイズ
*   @param[in]  bodyData        読み込みボディデータの格納先バッファ
*   @param[in]  bodySize        読み込むボディデータのバイトサイズ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*   @retval     ResultTargetNotFound        サムネイルファイルと関連付けられるセーブデータが作成されていません。
*
*   @pre
*       - buffer が有効なメモリを指している。
*       - headerSize が headerData のサイズ以下である。
*       - bodySize が bodyData のサイズ以下である。
*       - nn::account::Initialize() でアカウントシステムのライブラリが初期化済みである。
*
*   @details    ユーザーセーブデータに関連付くサムネイルファイルにデータを書き込みます。
*/
Result WriteSaveDataThumbnailFile(Bit64 saveDataOwnerId, const nn::account::Uid& user, const void* headerData, size_t headerSize, const void* bodyData, size_t bodySize) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      セーブデータサムネイルファイルにヘッダデータを書き込みます。
*
*   @param[in]  saveDataOwnerId セーブデータの所有者を示す ID
*   @param[in]  user            対象のユーザーを指す Uid
*   @param[in]  headerData      読み込みヘッダデータの格納先バッファ
*   @param[in]  headerSize      読み込むヘッダデータのバイトサイズ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*   @retval     ResultTargetNotFound        サムネイルファイルと関連付けられるセーブデータが作成されていません。
*
*   @pre
*       - buffer が有効なメモリを指している。
*       - headerSize が headerData のサイズ以下である。
*       - nn::account::Initialize() でアカウントシステムのライブラリが初期化済みである。
*
*   @details    ユーザーセーブデータに関連付くサムネイルファイルにヘッダデータを書き込みます。@n
*               headerSize が対象のサムネイルファイルに対する直前の nn::fs::WriteSaveDataThumbnailFile() 成功時の値と異なる場合
*               ボディデータが破壊され nn::fs::WriteSaveDataThumbnailFile() で書き直すまで
*               nn::fs::ReadSaveDataThumbnailFile() は ResultDataCorrupted を返します。
*               この状態のとき nn::fs::ReadSaveDataThumbnailFileHeader() は成功します。
*/
Result WriteSaveDataThumbnailFileHeader(Bit64 saveDataOwnerId, const nn::account::Uid& user, const void* headerData, size_t headerSize) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      セーブデータサムネイルファイルのデータを破壊します。
*
*   @param[in]  saveDataOwnerId セーブデータの所有者を示す ID
*   @param[in]  user            対象のユーザーを指す Uid
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*   @retval     ResultTargetNotFound        サムネイルファイルと関連付けられるセーブデータが作成されていません。
*
*   @pre
*       - nn::account::Initialize() でアカウントシステムのライブラリが初期化済みである。
*
*   @details    この API はデバッグ用途に限り使用可能です。@n
*               この API で破壊した対象は、 nn::fs::WriteSaveDataThumbnailFile() で書き直すまで、
*               nn::fs::ReadSaveDataThumbnailFile() および nn::fs::ReadSaveDataThumbnailFileHeader() は ResultDataCorrupted を返します。
*/
Result CorruptSaveDataThumbnailFileForDebug(Bit64 saveDataOwnerId, const nn::account::Uid& user) NN_NOEXCEPT;


//! @}

}} // namespace nn::fs
