﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
*   @file
*   @brief  fs 関連のエラーレポートを作成する API の実装
*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>

#include <nn/erpt.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_ErrorInfoPrivate.h>
#include <nn/fs/fs_ErrorReport.h>
#include <nn/fs/fs_Context.h>
#include <nn/fs/fs_GameCard.h>
#include <nn/fs/fs_MemoryReportInfo.h>
#include <nn/fs/fs_MmcPrivate.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/fs/fs_Utility.h>

#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_ApplicationManagerSystemApi.h>
#include <nn/ns/ns_SdCardApi.h>

namespace nn { namespace fs {

namespace detail {

inline Result SubmitStorageSizeInfo() NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    int64_t totalSize;
    int64_t freeSpaceSize;
    if( ns::GetStorageSize(&totalSize, &freeSpaceSize, ncm::StorageId::BuildInUser).IsSuccess() )
    {
        auto context = erpt::Context(erpt::NANDFreeSpaceInfo);
        NN_RESULT_DO(context.Add(erpt::NANDFreeSpace, static_cast<uint64_t>(freeSpaceSize)));
        NN_RESULT_DO(context.Add(erpt::NANDTotalSize, static_cast<uint64_t>(totalSize)));
        NN_RESULT_DO(context.SubmitContext());
    }
    if( ns::GetStorageSize(&totalSize, &freeSpaceSize, ncm::StorageId::SdCard).IsSuccess() )
    {
        auto context = erpt::Context(erpt::SDCardFreeSpaceInfo);
        NN_RESULT_DO(context.Add(erpt::SDCardFreeSpace, static_cast<uint64_t>(freeSpaceSize)));
        NN_RESULT_DO(context.Add(erpt::SdCardTotalSize, static_cast<uint64_t>(totalSize)));
        NN_RESULT_DO(context.SubmitContext());
    }
#elif( NN_BUILD_CONFIG_OS_WIN )
    // テスト用に適当な固定値を入れる。
    auto context = erpt::Context(erpt::NANDFreeSpaceInfo);
    NN_ABORT_UNLESS_RESULT_SUCCESS(context.Add(erpt::NANDFreeSpace, static_cast<uint64_t>(1234)));
    NN_ABORT_UNLESS_RESULT_SUCCESS(context.Add(erpt::NANDTotalSize, static_cast<uint64_t>(1234)));
    NN_ABORT_UNLESS_RESULT_SUCCESS(context.SubmitContext());
    NN_ABORT_UNLESS_RESULT_SUCCESS(context.ResetContext(erpt::SDCardFreeSpaceInfo));
    NN_ABORT_UNLESS_RESULT_SUCCESS(context.Add(erpt::SDCardFreeSpace, static_cast<uint64_t>(5678)));
    NN_ABORT_UNLESS_RESULT_SUCCESS(context.Add(erpt::SdCardTotalSize, static_cast<uint64_t>(5678)));
    NN_ABORT_UNLESS_RESULT_SUCCESS(context.SubmitContext());
#endif
    NN_RESULT_SUCCESS;
}

inline Result SetSdCardMountStatus() NN_NOEXCEPT
{
    // ns での SD カードの利用状態
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    auto context = erpt::Context(erpt::SdCardMountInfo);
    auto result = ns::CheckSdCardMountStatus();
    const int ResultLen = 16;
    char resultValueString[ResultLen];
    int len = util::SNPrintf(resultValueString, sizeof(resultValueString), "0x%08X", result.GetInnerValueForDebug());
    NN_RESULT_DO(context.Add(erpt::SdCardMountStatus, resultValueString, static_cast<uint32_t>(len)));
    if (ns::ResultSdCardMountUnexpected::Includes(result))
    {
        result = ns::GetLastSdCardMountUnexpectedResult();
        len = util::SNPrintf(resultValueString, sizeof(resultValueString), "0x%08X", result.GetInnerValueForDebug());
        NN_RESULT_DO(context.Add(erpt::SdCardMountUnexpectedResult, resultValueString, static_cast<uint32_t>(len)));
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(context.SubmitContext());
#elif( NN_BUILD_CONFIG_OS_WIN )
    // テスト用に適当な固定値を入れる。
    auto context = erpt::Context(erpt::SdCardMountInfo);
    const char* mountState = "None";
    NN_RESULT_DO(context.Add(erpt::SdCardMountStatus, mountState, static_cast<uint32_t>(strlen(mountState))));
    NN_ABORT_UNLESS_RESULT_SUCCESS(context.SubmitContext());
#endif
    NN_RESULT_SUCCESS;
}

// MMC 関連の情報を取得・設定する
inline Result SubmitMmcDetailInfo() NN_NOEXCEPT
{
    // MMC の CID 取得
    {
        uint8_t buffer[nn::fs::MmcCidSize] = {}; //16 バイト
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        // 実機版のときだけ正しい値を入れる
        if(nn::fs::GetMmcCid(buffer, sizeof(buffer)).IsSuccess())
#endif
            {
                auto context = erpt::Context(erpt::NANDCIDInfo);
                NN_RESULT_DO(context.Add(erpt::NANDCID, buffer, sizeof(buffer)));
                NN_RESULT_DO(context.SubmitContext());
            }
    }

    // MMC の SpeedMode 情報取得
    {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        nn::fs::MmcSpeedMode mmcSpeedMode;
        auto result = nn::fs::GetMmcSpeedMode(&mmcSpeedMode);
        auto context = erpt::Context(erpt::NANDSpeedModeInfo);
        if( result.IsSuccess() )
        {
            const char* speedModeName = "None";
            switch(mmcSpeedMode)
            {
            case nn::fs::MmcSpeedMode_Identification:
                speedModeName = "Identification";
                break;
            case nn::fs::MmcSpeedMode_LegacySpeed:
                speedModeName = "LegacySpeed";
                break;
            case nn::fs::MmcSpeedMode_HighSpeed:
                speedModeName = "HighSpeed";
                break;
            case nn::fs::MmcSpeedMode_Hs200:
                speedModeName = "Hs200";
                break;
            case nn::fs::MmcSpeedMode_Hs400:
                speedModeName = "Hs400";
                break;
            case nn::fs::MmcSpeedMode_Unknown:
                speedModeName = "Unknown";
                break;
            default:
                speedModeName = "UnDefined";
                break;
            }
            NN_RESULT_DO(context.Add(erpt::NANDSpeedMode, speedModeName, static_cast<uint32_t>(strlen(speedModeName))));
        }
        else
        {
            const int ResultLen = 18;
            char resultValueString[ResultLen];
            int len = util::SNPrintf(resultValueString, sizeof(resultValueString), "0x%08X", result.GetInnerValueForDebug());
            NN_RESULT_DO(context.Add(erpt::NANDSpeedMode, resultValueString, static_cast<uint32_t>(len)));
        }
        NN_RESULT_DO(context.SubmitContext());
#elif( NN_BUILD_CONFIG_OS_WIN )
        // テスト用に適当な固定値を入れる。
        const char* speedModeName = "None";
        auto context = erpt::Context(erpt::NANDSpeedModeInfo);
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.Add(erpt::NANDSpeedMode, speedModeName, static_cast<uint32_t>(strlen(speedModeName))));
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.SubmitContext());
#endif
    }

    // MMC の Extended Csd 取得
    {
        uint8_t buffer[nn::fs::MmcExtendedCsdSize] = {}; //512 バイト
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        // 実機版のときだけ正しい値を入れる
        if(nn::fs::GetMmcExtendedCsd(buffer, sizeof(buffer)).IsSuccess())
#endif
        {
            auto context = erpt::Context(erpt::NANDExtendedCsd);
            NN_RESULT_DO(context.Add(erpt::NANDPreEolInfo,            static_cast<uint32_t>(buffer[nn::fs::MmcExtendedCsdOffsetReEolInfo])));
            NN_RESULT_DO(context.Add(erpt::NANDDeviceLifeTimeEstTypA, static_cast<uint32_t>(buffer[nn::fs::MmcExtendedCsdOffsetDeviceLifeTimeEstTypA])));
            NN_RESULT_DO(context.Add(erpt::NANDDeviceLifeTimeEstTypB, static_cast<uint32_t>(buffer[nn::fs::MmcExtendedCsdOffsetDeviceLifeTimeEstTypB])));
            NN_RESULT_DO(context.SubmitContext());
        }
    }

    // MMC のパトロール情報取得
    {
        uint32_t mmcPatrolCount = 0;
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        // 実機版のときだけ正しい値を入れる
        if (nn::fs::GetMmcPatrolCount(&mmcPatrolCount).IsSuccess())
#endif
        {
            auto context = erpt::Context(erpt::NANDPatrolInfo);
            NN_RESULT_DO(context.Add(erpt::NANDPatrolCount, mmcPatrolCount));
            NN_RESULT_DO(context.SubmitContext());
        }
    }
    NN_RESULT_SUCCESS;
}

// MMC のエラー情報取得
inline Result SubmitMmcErrorInfo() NN_NOEXCEPT
{
    nn::fs::StorageErrorInfo storageErrorInfo;
    std::memset(&storageErrorInfo, 0x00, sizeof(nn::fs::StorageErrorInfo));
    char logBuffer[nn::erpt::ArrayBufferLength] = {};
    size_t logSize = 0;
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    // 実機版のときだけ正しい値を入れる
    if (nn::fs::GetAndClearMmcErrorInfo(&storageErrorInfo, &logSize, logBuffer, sizeof(logBuffer)).IsSuccess())
#endif
    {
        auto context = erpt::Context(erpt::NANDErrorInfo);
        NN_RESULT_DO(context.Add(erpt::NANDNumActivationFailures, storageErrorInfo.numActivationFailures));
        NN_RESULT_DO(context.Add(erpt::NANDNumActivationErrorCorrections, storageErrorInfo.numActivationErrorCorrections));
        NN_RESULT_DO(context.Add(erpt::NANDNumReadWriteFailures, storageErrorInfo.numReadWriteFailures));
        NN_RESULT_DO(context.Add(erpt::NANDNumReadWriteErrorCorrections, storageErrorInfo.numReadWriteErrorCorrections));
        NN_RESULT_DO(context.SubmitContext());
        if (logSize > 0)
        {
            NN_RESULT_DO(context.ResetContext(erpt::NANDDriverLog));
            NN_RESULT_DO(context.Add(erpt::NANDErrorLog, logBuffer, static_cast<uint32_t>(logSize)));
            NN_RESULT_DO(context.SubmitContext());
        }
    }
    NN_RESULT_SUCCESS;
}

// SD カード関連の情報を取得・設定する
inline Result SubmitSdCardDetailInfo() NN_NOEXCEPT
{
    // SD カード CID 取得
    {
        uint8_t buffer[nn::fs::SdCardCidSize] = {}; //16 バイト
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        // 実機版のときだけ正しい値を入れる
        if(nn::fs::GetSdCardCid(buffer, sizeof(buffer)).IsSuccess()) // 未挿入時は Failure が返る
#endif
        {
            auto context = erpt::Context(erpt::MicroSDCIDInfo);
            NN_RESULT_DO(context.Add(erpt::MicroSDCID, buffer, sizeof(buffer)));
            NN_RESULT_DO(context.SubmitContext());
        }
    }

    // SD カードの SpeedMode 取得
    {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        nn::fs::SdCardSpeedMode sdCardSpeedMode;
        auto result = nn::fs::GetSdCardSpeedMode(&sdCardSpeedMode);
        auto context = erpt::Context(erpt::MicroSDSpeedModeInfo);
        if( result.IsSuccess() ) // 未挿入時は Failure が返る
        {
            const char* speedModeName = "None";
            switch(sdCardSpeedMode)
            {
            case nn::fs::SdCardSpeedMode_Identification:
                speedModeName = "Identification";
                break;
            case nn::fs::SdCardSpeedMode_DefaultSpeed:
                speedModeName = "DefaultSpeed";
                break;
            case nn::fs::SdCardSpeedMode_HighSpeed:
                speedModeName = "HighSpeed";
                break;
            case nn::fs::SdCardSpeedMode_Sdr12:
                speedModeName = "Sdr12";
                break;
            case nn::fs::SdCardSpeedMode_Sdr25:
                speedModeName = "Sdr25";
                break;
            case nn::fs::SdCardSpeedMode_Sdr50:
                speedModeName = "Sdr50";
                break;
            case nn::fs::SdCardSpeedMode_Sdr104:
                speedModeName = "Sdr104";
                break;
            case nn::fs::SdCardSpeedMode_Ddr50:
                speedModeName = "Ddr50";
                break;
            case nn::fs::SdCardSpeedMode_Unknown:
                speedModeName = "Unknown";
                break;
            default:
                speedModeName = "UnDefined";
                break;
            }
            NN_RESULT_DO(context.Add(erpt::MicroSDSpeedMode, speedModeName, static_cast<uint32_t>(strlen(speedModeName))));
        }
        else
        {
            const int ResultLen = 18;
            char resultValueString[ResultLen];
            int len = util::SNPrintf(resultValueString, sizeof(resultValueString), "0x%08X", result.GetInnerValueForDebug());
            NN_RESULT_DO(context.Add(erpt::MicroSDSpeedMode, resultValueString, static_cast<uint32_t>(len)));
        }
        NN_RESULT_DO(context.SubmitContext());
#elif( NN_BUILD_CONFIG_OS_WIN )
        // テスト用に適当な固定値を入れる。
        const char* speedModeName = "None";
        auto context = erpt::Context(erpt::MicroSDSpeedModeInfo);
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.Add(erpt::MicroSDSpeedMode, speedModeName, static_cast<uint32_t>(strlen(speedModeName))));
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.SubmitContext());
#endif
    }

    // SD カード各サイズ 取得
    {
        int64_t userAreaSize = 0;
        int64_t protectedAreaSize = 0;
        nn::Result resultUser = nn::ResultSuccess();
        nn::Result resultProt = nn::ResultSuccess();
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        // 実機版のときだけ正しい値を入れる
        resultUser = nn::fs::GetSdCardUserAreaSize(&userAreaSize);
        resultProt = nn::fs::GetSdCardProtectedAreaSize(&protectedAreaSize);
#endif
        if(resultUser.IsSuccess() || resultProt.IsSuccess()) // 未挿入時は Failure が返る
        {
            auto context = erpt::Context(erpt::SdCardSizeSpec);
            if(resultUser.IsSuccess())
            {
                NN_RESULT_DO(context.Add(erpt::SdCardUserAreaSize, userAreaSize));
            }
            if(resultProt.IsSuccess())
            {
                NN_RESULT_DO(context.Add(erpt::SdCardProtectedAreaSize, protectedAreaSize));
            }
            NN_RESULT_DO(context.SubmitContext());
        }
    }
    NN_RESULT_SUCCESS;
}

// SD カードのエラー情報取得
inline Result SubmitSdCardErrorInfo() NN_NOEXCEPT
{
    nn::fs::StorageErrorInfo storageErrorInfo;
    std::memset(&storageErrorInfo, 0x00, sizeof(nn::fs::StorageErrorInfo));
    char logBuffer[nn::erpt::ArrayBufferLength] = {};
    size_t logSize = 0;
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    // 実機版のときだけ正しい値を入れる
    if(nn::fs::GetAndClearSdCardErrorInfo(&storageErrorInfo, &logSize, logBuffer, sizeof(logBuffer)).IsSuccess())
#endif
    {
        auto context = erpt::Context(erpt::SdCardErrorInfo);
        NN_RESULT_DO(context.Add(erpt::SdCardNumActivationFailures, storageErrorInfo.numActivationFailures));
        NN_RESULT_DO(context.Add(erpt::SdCardNumActivationErrorCorrections, storageErrorInfo.numActivationErrorCorrections));
        NN_RESULT_DO(context.Add(erpt::SdCardNumReadWriteFailures, storageErrorInfo.numReadWriteFailures));
        NN_RESULT_DO(context.Add(erpt::SdCardNumReadWriteErrorCorrections, storageErrorInfo.numReadWriteErrorCorrections));
        NN_RESULT_DO(context.SubmitContext());
        if(logSize > 0)
        {
            NN_RESULT_DO(context.ResetContext(erpt::SdCardDriverLog));
            NN_RESULT_DO(context.Add(erpt::SdCardErrorLog, logBuffer, static_cast<uint32_t>(logSize)));
            NN_RESULT_DO(context.SubmitContext());
        }
    }
    NN_RESULT_SUCCESS;
}

inline Result SubmitGameCardDetailInfo() NN_NOEXCEPT
{
    auto context = nn::erpt::Context(nn::erpt::GameCardCIDInfo);
    // ゲームカード CID 情報取得
    {
        uint8_t buffer[nn::fs::GameCardCidSize] = {}; //16 バイト
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    // 実機版のときだけ正しい値を入れる
        if (nn::fs::IsGameCardInserted() && nn::fs::GetGameCardCid(buffer, sizeof(buffer)).IsSuccess())
#endif
        {
            NN_RESULT_DO(context.Add(nn::erpt::GameCardCID, buffer, sizeof(buffer)));
        }
    }

    // ゲームカード DeviceID 情報取得
    {
        uint8_t buffer[nn::fs::GameCardDeviceIdSize] = {}; //16 バイト
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    // 実機版のときだけ正しい値を入れる
        if (nn::fs::IsGameCardInserted() && nn::fs::GetGameCardDeviceId(buffer, sizeof(buffer)).IsSuccess())
#endif
        {
            NN_RESULT_DO(context.Add(nn::erpt::GameCardDeviceId, buffer, sizeof(buffer)));
        }
    }
    NN_RESULT_DO(context.SubmitContext());

    NN_RESULT_SUCCESS;
}

inline Result SubmitGameCardErrorInfo() NN_NOEXCEPT
{
    nn::fs::GameCardErrorReportInfo gameCardErrorReportInfo;
    std::memset(&gameCardErrorReportInfo, 0x00, sizeof(nn::fs::GameCardErrorReportInfo));
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    // 実機版のときだけ正しい値を入れる
    // GameCard アクセスに失敗しても ABORT はしない
    if(nn::fs::GetGameCardErrorReportInfo(&gameCardErrorReportInfo).IsSuccess())
#endif
    {
        auto context = nn::erpt::Context(nn::erpt::GameCardErrorInfo);
        NN_RESULT_DO(context.Add(nn::erpt::GameCardCrcErrorCount, static_cast<uint32_t>(gameCardErrorReportInfo.gameCardCrcErrorNum)));
        NN_RESULT_DO(context.Add(nn::erpt::GameCardAsicCrcErrorCount, static_cast<uint32_t>(gameCardErrorReportInfo.asicCrcErrorNum)));
        NN_RESULT_DO(context.Add(nn::erpt::GameCardRefreshCount, static_cast<uint32_t>(gameCardErrorReportInfo.refreshNum)));
        NN_RESULT_DO(context.Add(nn::erpt::GameCardReadRetryCount, static_cast<uint32_t>(gameCardErrorReportInfo.retryLimitOutNum)));
        NN_RESULT_DO(context.Add(nn::erpt::GameCardTimeoutRetryErrorCount, static_cast<uint32_t>(gameCardErrorReportInfo.timeoutRetryNum)));
        NN_RESULT_DO(context.Add(nn::erpt::GameCardInsertionCount, static_cast<uint32_t>(gameCardErrorReportInfo.insertionCount)));
        NN_RESULT_DO(context.Add(nn::erpt::GameCardRemovalCount, static_cast<uint32_t>(gameCardErrorReportInfo.removalCount)));
        NN_RESULT_DO(context.Add(nn::erpt::GameCardAsicInitializeCount, gameCardErrorReportInfo.initializeCount));
        NN_RESULT_DO(context.Add(nn::erpt::GameCardAsicReinitializeCount, gameCardErrorReportInfo.asicReinitializeNum));
        NN_RESULT_DO(context.Add(nn::erpt::GameCardAsicReinitializeFailureCount, gameCardErrorReportInfo.asicReinitializeFailureNum));
        NN_RESULT_DO(context.Add(nn::erpt::GameCardAsicReinitializeFailureDetail, gameCardErrorReportInfo.asicReinitializeFailureDetail));
        NN_RESULT_DO(context.Add(nn::erpt::GameCardRefreshSuccessCount, gameCardErrorReportInfo.refreshSucceededCount));
        NN_RESULT_DO(context.Add(nn::erpt::GameCardAwakenCount, gameCardErrorReportInfo.awakenCount));
        NN_RESULT_DO(context.Add(nn::erpt::GameCardAwakenFailureCount, gameCardErrorReportInfo.awakenFailureNum));
        NN_RESULT_DO(context.Add(nn::erpt::GameCardReadCountFromInsert, gameCardErrorReportInfo.readCountFromInsert));
        NN_RESULT_DO(context.Add(nn::erpt::GameCardReadCountFromAwaken, gameCardErrorReportInfo.readCountFromAwaken));
        NN_RESULT_DO(context.Add(nn::erpt::GameCardLastReadErrorPageAddress, gameCardErrorReportInfo.lastReadErrorPageAddress));
        NN_RESULT_DO(context.Add(nn::erpt::GameCardLastReadErrorPageCount, gameCardErrorReportInfo.lastReadErrorPageCount));
        NN_RESULT_DO(context.SubmitContext());
    }
    NN_RESULT_SUCCESS;
}

// FS のエラー情報取得
inline Result SubmitFileSystemErrorInfo() NN_NOEXCEPT
{
    nn::fs::FileSystemProxyErrorInfo fileSystemProxyErrorInfo;
    std::memset(&fileSystemProxyErrorInfo, 0x00, sizeof(nn::fs::FileSystemProxyErrorInfo));
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    // 実機版のときだけ正しい値を入れる
    if(nn::fs::GetAndClearFileSystemProxyErrorInfo(&fileSystemProxyErrorInfo).IsSuccess())
#endif
    {
        auto context = erpt::Context(erpt::FsProxyErrorInfo);
        NN_RESULT_DO(context.Add(erpt::FsRemountForDataCorruptCount, fileSystemProxyErrorInfo.romFsRemountForDataCorruptionCount));
        NN_RESULT_DO(context.Add(erpt::FsRemountForDataCorruptRetryOutCount, fileSystemProxyErrorInfo.romFsUnrecoverableDataCorruptionByRemountCount));
        NN_RESULT_DO(context.Add(erpt::FatFsError, fileSystemProxyErrorInfo.fatFsError.error));
        NN_RESULT_DO(context.Add(erpt::FatFsExtraError, fileSystemProxyErrorInfo.fatFsError.extraError));
        NN_RESULT_DO(context.Add(erpt::FatFsErrorDrive, fileSystemProxyErrorInfo.fatFsError.driveId));
        NN_RESULT_DO(context.Add(erpt::FatFsErrorName, fileSystemProxyErrorInfo.fatFsError.name, nn::fat::FatErrorNameMaxLength));
        NN_RESULT_DO(context.Add(erpt::FsRecoveredByInvalidateCacheCount, fileSystemProxyErrorInfo.romFsRecoveredByInvalidateCacheCount));
        NN_RESULT_DO(context.Add(erpt::FsSaveDataIndexCount, fileSystemProxyErrorInfo.saveDataIndexCount));
        NN_RESULT_DO(context.SubmitContext());
    }
    NN_RESULT_SUCCESS;
}

// メモリ枯渇状態情報取得
inline Result SubmitMemoryReportInfo() NN_NOEXCEPT
{
    nn::fs::MemoryReportInfo memoryReportInfo;
    std::memset(&memoryReportInfo, 0x00, sizeof(nn::fs::MemoryReportInfo));
    if( nn::fs::GetAndClearMemoryReportInfo(&memoryReportInfo).IsSuccess() )
    {
        auto context = erpt::Context(erpt::FsMemoryInfo);
        NN_RESULT_DO(context.Add(erpt::FsPooledBufferPeakFreeSize, memoryReportInfo.pooledBufferPeakFreeSize));
        NN_RESULT_DO(context.Add(erpt::FsPooledBufferRetriedCount, memoryReportInfo.pooledBufferRetriedCount));
        NN_RESULT_DO(context.Add(erpt::FsPooledBufferReduceAllocationCount, memoryReportInfo.pooledBufferReduceAllocationCount));
        NN_RESULT_DO(context.Add(erpt::FsBufferManagerPeakFreeSize, memoryReportInfo.bufferManagerPeakFreeSize));
        NN_RESULT_DO(context.Add(erpt::FsBufferManagerRetriedCount, memoryReportInfo.bufferManagerRetriedCount));
        NN_RESULT_DO(context.Add(erpt::FsExpHeapPeakFreeSize, memoryReportInfo.expHeapPeakFreeSize));
        NN_RESULT_DO(context.Add(erpt::FsBufferPoolPeakFreeSize, memoryReportInfo.bufferPoolPeakFreeSize));
        NN_RESULT_DO(context.Add(erpt::FsPatrolReadAllocateBufferSuccessCount, memoryReportInfo.patrolReadAllocateBufferSuccessCount));
        NN_RESULT_DO(context.Add(erpt::FsPatrolReadAllocateBufferFailureCount, memoryReportInfo.patrolReadAllocateBufferFailureCount));
        NN_RESULT_DO(context.Add(erpt::FsBufferManagerPeakTotalAllocatableSize, memoryReportInfo.bufferManagerPeakTotalAllocatableSize));
        NN_RESULT_DO(context.SubmitContext());
    }
    NN_RESULT_SUCCESS;
}

// NAND 関連の情報をエラーレポートに残す
inline void SetMmcStorageInfo() NN_NOEXCEPT
{
    SubmitMmcDetailInfo();
    SubmitMmcErrorInfo();
}

// SD カード 関連の情報をエラーレポートに残す
inline void SetSdCardStorageInfo() NN_NOEXCEPT
{
    SubmitSdCardDetailInfo();
    SubmitSdCardErrorInfo();
}

// ゲームカード 関連の情報をエラーレポートに残す
inline void SetGameCardStorageInfo() NN_NOEXCEPT
{
    SubmitGameCardDetailInfo();
    SubmitGameCardErrorInfo();
}

inline void ClearErptContext(erpt::CategoryId category) NN_NOEXCEPT
{
    auto context = erpt::Context(category);
    NN_ABORT_UNLESS_RESULT_SUCCESS(context.SubmitContext());
}

} // namespace nn::fs::detail

/**
*   @brief      fs 関連のエラーレポート情報を送信します。
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*/
inline Result SetErrorReportFileSystemInfo() NN_NOEXCEPT
{
    // 失敗時 fs shim で強制アボートにならないように一時的に抑制する
    FsContext fsContext(
        [](Result result)
        {
            NN_UNUSED(result);
            return AbortSpecifier::ReturnResult;
        }
    );
    // 生存する間、カレントスレッドでのファイルシステムコンテキストを設定するユーティリティクラス
    ScopedFsContextSetter scopedContext(fsContext);
    // エラーレポートに登録
    detail::SetMmcStorageInfo();
    detail::SetSdCardStorageInfo();
    detail::SetGameCardStorageInfo();
    detail::SubmitFileSystemErrorInfo();
    detail::SubmitMemoryReportInfo();
    NN_RESULT_SUCCESS;
}

/**
*   @brief      ns 関連のエラーレポート情報を送信します。
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*/
inline Result SetErrorReportNsInfo() NN_NOEXCEPT
{
    // 失敗時 fs shim で強制アボートにならないように一時的に抑制する
    FsContext fsContext(
        [](Result result)
        {
            NN_UNUSED(result);
            return AbortSpecifier::ReturnResult;
        }
    );
    // 生存する間、カレントスレッドでのファイルシステムコンテキストを設定するユーティリティクラス
    ScopedFsContextSetter scopedContext(fsContext);
    // エラーレポートに登録
    detail::SubmitStorageSizeInfo();
    detail::SetSdCardMountStatus();
    NN_RESULT_SUCCESS;
}

/**
*   @brief      fs 関連のエラーレポート情報をクリアします。
*/
inline void ClearErrorReportFileSystemInfo() NN_NOEXCEPT
{
    detail::ClearErptContext(erpt::NANDSpeedModeInfo);
    detail::ClearErptContext(erpt::MicroSDSpeedModeInfo);

    detail::ClearErptContext(erpt::NANDCIDInfo);
    detail::ClearErptContext(erpt::NANDExtendedCsd);
    detail::ClearErptContext(erpt::NANDPatrolInfo);

    detail::ClearErptContext(erpt::MicroSDCIDInfo);
    detail::ClearErptContext(erpt::SdCardSizeSpec);

    detail::ClearErptContext(erpt::GameCardCIDInfo);

    detail::ClearErptContext(erpt::NANDErrorInfo);
    detail::ClearErptContext(erpt::NANDDriverLog);

    detail::ClearErptContext(erpt::SdCardErrorInfo);
    detail::ClearErptContext(erpt::SdCardDriverLog);

    detail::ClearErptContext(erpt::GameCardErrorInfo);

    detail::ClearErptContext(erpt::FsProxyErrorInfo);

    detail::ClearErptContext(erpt::FsMemoryInfo);
}

/**
*   @brief      ns 関連のエラーレポート情報をクリアします。
*/
inline void ClearErrorReportNsInfo() NN_NOEXCEPT
{
    detail::ClearErptContext(erpt::NANDFreeSpaceInfo);
    detail::ClearErptContext(erpt::SDCardFreeSpaceInfo);
    detail::ClearErptContext(erpt::SdCardMountInfo);
}

}} // namespace nn::fs
