﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
*   @file
*   @brief  ファイルシステム に関する API の宣言
*
*   :include nn/fs.h
*/


#pragma once


/**
*   @namespace  nn::fs
*   @brief      ファイルシステムライブラリの名前空間です。
*
*               ### 機能概要
*               FS ライブラリでは、セーブデータやホスト PC 上ファイルシステムなど、様々なファイルシステムに対して操作を行う手段を提供します。@n
*               ディレクトリを開くなどの基本的な機能は各種ファイルシステムで共通ですが、ファイルシステム固有の機能や制限などもあり、これらはファイルシステムの属性として別途定義されます。@n
*
*               #### 共通の事前条件
*               ファイル、ディレクトリに対して操作を行う関数については、対象のファイル、ディレクトリを表すハンドルが正常に取得されている必要があります。@n
*               また特に記載のない限り、ポインタ引数は有効なメモリを指している必要があります。@n
*
*               #### Result について
*               マウント対象のファイルシステムの種類によっては、fs 以外の Result も返る可能性があります。@n
*
*               #### @anchor memory_allocation メモリ確保について
*               戻り値一覧に ResultAllocationMemoryFailed が挙げられている API ( Mount 系 API, OpenFile(), OpenDirectory() ) は内部でメモリ確保を行います。
*               デフォルトではメモリ確保に malloc(), free() が使用されます。
*               malloc(), free() については nn::init::InitializeAllocator() も併せて参照してください。
*               @n
*               nn::fs::SetAllocator() をあらかじめ呼び出すことで、メモリ確保に使用されるアロケータを指定することができます。
*               設定するアロケータは 8 バイトアライメントのアドレスを返す必要があります。
*               @n
*               確保されたメモリは、対応する解放 API ( Unmount(), CloseFile(), CloseDirectory() ) を呼び出すことで解放されます。
*               ファイルおよびディレクトリが一つも開かれておらず、ファイルシステムが一つもマウントされていない状態では、
*               アロケータからは一切のメモリが確保されていない状態になること（確保されたメモリがすべて解放されていること）が保証されます。
*               @n
*
*               メモリ使用量の目安は以下の通りです。これらの値は暫定のものであり、将来的に増減する可能性があります。@n
*               | API | メモリ使用量の目安 |
*               | ---- | ---- |
*               | Mount 系関数 | 同時マウント数 × 5 KB 程度 |
*               | OpenFile | 同時オープン数 × 100 B 程度 |
*               | OpenDirectory | 同時オープン数 × 2 KB 程度 |
*               @n
*
*               ライブラリのような不特定のプログラムから参照される可能性がある関数群でファイルシステムの機能を利用する場合には、
*               nn::fs::SetAllocator() で事前に指定したアロケータまたは malloc() で十分なメモリが確保できる必要があることを関数群の利用者に対して明示してください。
*
*               #### スレッドセーフ性について
*               FS ライブラリの API には、複数同時呼び出しが可能なものと、そうではなく排他制御が必要なものがあります。@n
*               @n
*               ファイルシステム関連 API 及び OpenFile()、 OpenDirectory() は、複数同時に呼び出された場合でも正しく動作します。@n
*               ファイル関連 API、ディレクトリ関連 API のうち、ハンドルを指定して操作を行う API については、以下の表の通りです。@n
*
*               | API の種類 | 同一のハンドルが対象 | 別々のハンドルが対象 |
*               | ---- | ---- | ---- |
*               | ファイル関連 API で、読み込み操作のみを行うもの (ReadFile(), GetFileSize(), GetFileOpenMode())  | 複数同時呼び出しが可能 | 複数同時呼び出しが可能 |
*               | ファイル関連 API で、読み込み以外の操作も行うもの (OpenFile() と上記 API 以外)  | 排他的な呼び出しが必要 | 複数同時呼び出しが可能 |
*               | ディレクトリ関連 API (OpenDirectory(), GetDirectoryEntryCount() 以外) | 排他的な呼び出しが必要 | 複数同時呼び出しが可能 |
*
*               上記以外のその他の API の呼び出しや、複数同時呼び出しが可能な API とそうでない API を組み合わせた呼び出しは、特に明記されていない限り排他的に行う必要があります。@n
*               例えば、同一のハンドルに対して複数のスレッドで排他せずに WriteFile() や ReadFile() を行うと、タイミングに依存して ReadFile() で得られるデータが変わります。
*               例えば、同一のハンドルに対して複数のスレッドで排他せずに ReadDirectory() を行うと、それぞれのスレッドのディレクトリ列挙処理が衝突するため、挙動は保証されません。@n
*   @platformbegin{NX}
*               本体保存メモリー、ゲームカード、SD カードへの通信経路はそれぞれ一つのため、
*               アプリケーションから同じ記録メディアに配置されるデータを複数同時にアクセスしても内部的な処理はブロッキングされます。@n
*               パフォーマンスの最適化方法は、NintendoSDK ドキュメントの FS ライブラリ @confluencelink{185130689, パフォーマンス} を参照してください。@n
*   @platformend
*
*               ### 用語
*               #### ファイルシステム
*               FS ライブラリにおける、操作対象の基本単位で、ディレクトリやファイルが格納されたものです。@n
*               ファイルシステムをマウントすることで、そのファイルシステムが所有するディレクトリ、ファイルに対して操作を行うことが可能になります。@n
*
*               #### エントリ
*               ディレクトリとファイルを総称したものです。@n
*
*               #### @anchor mount_name マウント名
*               ファイルシステムを利用するためにマウントした際に、識別子としてユーザーが与える文字列です。@n
*               マウント操作の後は、マウント名を使用してファイルシステムを指定してアクセスします。@n
*               以下の全ての条件を満たしているマウント名を有効なマウント名とします@n
*               - 終端の NULL 文字を含めず 1 バイト以上 nn::fs::MountNameLengthMax バイト以下の文字列である。
*               -  '@' 以外の文字で始まる。
*               -  ':' および '/' を含まない。
*               - マウントしている他のファイルシステムのマウント名に使用されていない。
*               - アルファベット一文字でない。
*
*               マウント名を指定する API には有効なマウント名を渡す必要があります。@n
*               エンコーディングは UTF-8 です。@n
*
*               #### @anchor entry_path パス
*               エントリを一意に定めるための識別子です。@n
*               FS ライブラリでは文字列を使用します。@n
*               例： "MountName:/path/to/file" @n
*               以下の全ての条件を満たしているパスを有効なパスとします。@n
*               - 「マウント名 + ":/"」または「マウント名 + ":\"」で始まる。
*               - マウント名が有効なマウント名である。
*               - マウント名がマウント済みのものである。
*               - ディレクトリの区切り文字が '/' または '\'。
*               - エントリ名に '<'、'>'、'*'、'?'、':'、'|' を使用していない。
*               - 文字列長が 260 文字以下である。
*               - 終端の NULL 文字を含めず 1 バイト以上 nn::fs::EntryNameLengthMax バイト以下の文字列である。
*
*               パスを指定する API には有効なパスを渡す必要があります。@n
*               エンコーディングは UTF-8 です。@n
*               ホスト PC 上のパスを表す場合でも、同じ区切り文字・エンコーディングを使用します。@n
*   @platformbegin{NX}
*               SD カードファイルシステムで使用するパスには特有の制約があります。NintendoSDK ドキュメントの @confluencelink{112886110, SD カード} を参照してください。@n
*   @platformend
*
*               <h3>@anchor supported_filesystem サポートされているファイルシステム</h3>
*               各ファイルシステムの詳細は NintendoSDK ドキュメントの FS ライブラリ @confluencelink{107320200, 機能説明} を参照してください。
*               - ROM ファイルシステム@n
*               リソースデータを扱う読み取り専用のファイルシステムです。@n
*               nn::fs::MountRom() でマウントできます。@n
*               ジャーナリング機能、マルチコミット機能は無効です。@n
*               - セーブデータファイルシステム ※@n
*               セーブデータを扱うファイルシステムです。@n
*               nn::fs::MountSaveData(), nn::fs::MountSaveDataReadOnly(), nn::fs::MountSaveDataForDebug() でマウントできます。@n
*               ジャーナリング機能、マルチコミット機能は有効です。@n
*               - ホストファイルシステム@n
*               ホスト PC 上のファイルシステムを操作できるファイルシステムです。@n
*               nn::fs::MountHost(), nn::fs::MountHostRoot() でマウントできます。@n
*               ジャーナリング機能、マルチコミット機能は無効です。@n
*               開発時のみ使用することができます。
*
*   @platformbegin{NX}
*               - 追加コンテンツファイルシステム@n
*               追加コンテンツを扱う読み込み専用のファイルシステムです。@n
*               nn::fs::MountAddOnContent() でマウントできます。@n
*               ジャーナリング機能、マルチコミット機能は無効です。@n
*               - 一時ストレージファイルシステム ※@n
*               アプリケーションが一時的にファイルを保存するためのファイルシステムです。@n
*               nn::fs::MountTemporaryStorage() でマウントできます。@n
*               ジャーナリング機能、マルチコミット機能は無効です。@n
*               一時ストレージを利用するには事前の申請が必要です。
*               - キャッシュストレージファイルシステム ※@n
*               アプリケーションが再生成・再取得可能なデータを保存するためのファイルシステムです。@n
*               nn::fs::MountCacheStorage() でマウントできます。@n
*               ジャーナリング機能は有効ですが、マルチコミット機能は無効です。@n
*               キャッシュストレージを利用するには事前の申請が必要です。
*               - SD カードファイルシステム@n
*               SD カード 上のファイルシステムを操作できるファイルシステムです。@n
*               nn::fs::MountSdCardForDebug() でマウントできます。@n
*               ジャーナリング機能、マルチコミット機能は無効です。@n
*               開発時のみ使用することができます。
*
*   @platformend
*               <h4>同時にマウントできる数の上限</h4>
*               ファイルシステムによって同時にマウントできる数に制限があります。
*               以下に記載されていないファイルシステムはマウント数にチェックはありません。
*               - 上記で ※ が付けられているファイルシステム@n
*               同じ対象をマウント名を変えて同時にマウントすることは出来ません。
*               異なる対象を同時にマウントできる数の上限は ※ が付けられているファイルシステムのマウント数を合わせて 10 です。@n
*               nn::fs::IsSaveDataExisting() も実行中はマウントとしてカウントされます。
*               - ROM ファイルシステム@n
*               マウント名を変えて同時にマウントできる数の上限は 10 です。@n
*               nn::fs::QueryMountRomCacheSize(), nn::fs::CanMountRomForDebug() も実行中はマウントとしてカウントされます。
*
*   @platformbegin{NX}
*               - 追加コンテンツファイルシステム@n
*               同じ対象をマウント名を変えて同時にマウントすることができ、同時にマウントできる数の上限は全ての追加コンテンツのマウント数を合わせて 128 です。@n
*               nn::fs::QueryMountAddOnContentCacheSize() も実行中はマウントとしてカウントされます。
*
*   @platformend
*               <h4>同時にオープンできる数の上限</h4>
*               ファイルシステムによってファイルとディレクトリを同時にオープンできる数に制限があります。
*               以下に記載されていないファイルシステムはオープン数にチェックはありません。
*               - 上記で ※ が付けられているファイルシステム@n
*                 ファイルとディレクトリを同時にオープンできる数の上限は ※ が付けられているファイルシステムにおけるファイルとディレクトリのオープン数を全て合せて 256 です。@n
*                 同じ対象を重ねてオープンする場合もオープン数がカウントされます。@n
*                 nn::fs::EnableIndividualFileDataCache() も実行中はオープンとしてカウントされます。
*
*/

#include <nn/fs_Base.h>
#include <nn/fs/fs_SaveData.h>
#include <nn/fs/fs_TimeStampForDebug.h>
#if defined(NN_BUILD_CONFIG_SPEC_NX)
    #include <nn/fs/fs_AddOnContent.h>
    #include <nn/fs/fs_CacheStorageWithIndex.h>
#endif
