﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/friends/friends_TypesAdmin.h>
#include <nn/friends/friends_Macros.h>
#include <nn/friends/friends_ProfileExtra.h>
#include <nn/friends/friends_AsyncContext.h>

namespace nn { namespace friends {

/*!
    @brief      ユーザー設定を扱うクラスです。

    @details
                本クラスは、プレゼンスの公開範囲やプレイログの管理等、ユーザーごとの設定情報を管理します。

                自分の設定はネットワーク未接続時にキャッシュを参照することはできますが、更新することはできません。@n
                別デバイス（スマートデバイス含む）で設定を変更した場合、ユーザーの選択時に本体ストレージ上のキャッシュは更新されます。
*/
class UserSetting
{
public:
    /*!
        @brief      コンストラクタです。
    */
    UserSetting() NN_NOEXCEPT;

    /*!
        @brief      初期化します。

        @param[in]  uid ユーザーアカウント。

        @return     処理結果。

        @pre
            - 指定したユーザーアカウントが Open 状態である。
    */
    nn::Result Initialize(const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      プレゼンスの公開範囲を取得します。

        @return     プレゼンスの公開範囲。

        @details
                    本関数は本体ストレージ上のキャッシュから設定を取得します。@n
                    そのため、ネットワーク未接続時でも設定を取得することができます。
    */
    PresencePermission GetPresencePermission() const NN_NOEXCEPT;

    /*!
        @brief      プレゼンスの公開範囲を変更します。

        @param[out] outAsync    非同期コンテキスト。
        @param[in]  newValue    新しい設定値。

        @return     処理結果。

        @pre
            - outAsync != nullptr
            - NIFM ライブラリを初期化している。

        @details
                    本関数は通信処理を行うため、事前にインターネット利用要求を行い、それが受理されている必要があります。@n
                    通信処理のエラーハンドリングは、 @ref AsyncContext::GetResult を参照してください。
    */
    nn::Result ChangePresencePermission(AsyncContext* outAsync, PresencePermission newValue) NN_NOEXCEPT;

    /*!
        @brief      フレンド申請の受信設定を取得します。

        @return     フレンド申請の受信設定。

        @details
                    本関数は本体ストレージ上のキャッシュから設定を取得します。@n
                    そのため、ネットワーク未接続時でも設定を取得することができます。
    */
    bool GetFriendRequestReception() const NN_NOEXCEPT;

    /*!
        @brief      フレンド申請の受信設定を変更します。

        @param[out] outAsync    非同期コンテキスト。
        @param[in]  newValue    新しい設定値。

        @return     処理結果。

        @pre
            - outAsync != nullptr
            - NIFM ライブラリを初期化している。

        @details
                    本関数は通信処理を行うため、事前にインターネット利用要求を行い、それが受理されている必要があります。@n
                    通信処理のエラーハンドリングは、 @ref AsyncContext::GetResult を参照してください。
    */
    nn::Result ChangeFriendRequestReception(AsyncContext* outAsync, bool newValue) NN_NOEXCEPT;

    /*!
        @brief      フレンドコードを取得します。

        @return     フレンドコード。

        @details
                    本関数は本体ストレージ上のキャッシュからフレンドコードを取得します。@n
                    そのため、ネットワーク未接続時でもフレンドコードを取得することができます。

                    フレンドコードが未発効の場合、本関数は空文字列を返します。
    */
    const FriendCode& GetFriendCode() NN_NOEXCEPT;

    /*!
        @brief      フレンドコードの次回発行可能時刻を取得します。

        @return     フレンドコードの次回発行可能時刻。

        @details
                    本関数は本体ストレージ上のキャッシュからフレンドコードの次回発行可能時刻を取得します。@n
                    そのため、ネットワーク未接続時でもフレンドコードの次回発行可能時刻を取得することができます。

                    フレンドコードが未発効の場合、本関数は 0 を返します。
    */
    const nn::time::PosixTime& GetFriendCodeNextIssuableTime() NN_NOEXCEPT;

    /*!
        @brief      フレンドコードを発行します。

        @param[out] outAsync    非同期コンテキスト。

        @return     処理結果。

        @pre
            - outAsync != nullptr
            - NIFM ライブラリを初期化している。

        @details
                    現在時刻が次回発行可能時刻を過ぎていない場合、非同期処理の結果は ResultDisallowFriendCodeIssue になります。

                    次回発行可能時刻を超えている場合、フレンドコードを（再）発行することができます。@n
                    フレンドコードを既に発行している場合、現在のフレンドコードは無効化して利用されないようにします。

                    本関数は通信処理を行うため、事前にインターネット利用要求を行い、それが受理されている必要があります。@n
                    通信処理のエラーハンドリングは、 @ref AsyncContext::GetResult を参照してください。
    */
    nn::Result IssueFriendCode(AsyncContext* outAsync) NN_NOEXCEPT;

    /*!
        @brief      他者に見える自分の拡張プロフィールを取得します。

        @return     拡張プロフィール。

        @details
                    本関数は本体ストレージ上のキャッシュから拡張プロフィールを取得します。@n
                    そのため、ネットワーク未接続時でも拡張プロフィールを取得することができます。
    */
    const ProfileExtra& GetExtra() const NN_NOEXCEPT;

    /*!
        @brief      拡張プロフィール内のプレイログの公開範囲を取得します。

        @return     プレイログの公開範囲。

        @details
                    本関数は本体ストレージ上のキャッシュから拡張プロフィール内のプレイログを取得します。@n
                    そのため、ネットワーク未接続時でも拡張プロフィール内のプレイログを取得することができます。
    */
    ExtraPermission GetPlayLogPermission() const NN_NOEXCEPT;

    /*!
        @brief      拡張プロフィール内のプレイログの公開範囲を変更します。

        @param[out] outAsync    非同期コンテキスト。
        @param[in]  newValue    新しい設定値。

        @return     処理結果。

        @pre
            - outAsync != nullptr
            - NIFM ライブラリを初期化している。

        @details
                    本関数は通信処理を行うため、事前にインターネット利用要求を行い、それが受理されている必要があります。@n
                    通信処理のエラーハンドリングは、 @ref AsyncContext::GetResult を参照してください。
    */
    nn::Result ChangePlayLogPermission(AsyncContext* outAsync, ExtraPermission newValue) NN_NOEXCEPT;

    /*!
        @brief      拡張プロフィール内のプレイログをすべて消去します。

        @param[out] outAsync    非同期コンテキスト。

        @return     処理結果。

        @pre
            - outAsync != nullptr
            - NIFM ライブラリを初期化している。

        @details
                    本関数は通信処理を行うため、事前にインターネット利用要求を行い、それが受理されている必要があります。@n
                    通信処理のエラーハンドリングは、 @ref AsyncContext::GetResult を参照してください。
    */
    nn::Result ClearPlayLog(AsyncContext* outAsync) NN_NOEXCEPT;

private:
    //
    NN_FRIENDS_CLASS_STORAGE(2048);
};

/*!
    @brief      フレンドごとの設定を扱うクラスです。
*/
class FriendSetting
{
public:
    /*!
        @brief      コンストラクタです。
    */
    FriendSetting() NN_NOEXCEPT;

    /*!
        @brief      初期化します。

        @param[in]  uid         ユーザーアカウント。
        @param[in]  accountId   フレンドのネットワークサービスアカウント ID。

        @return     処理結果。
    */
    nn::Result Initialize(const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT;

    /*!
        @brief      新規成立フラグを取得します。

        @return     新規成立フラグ。
    */
    bool GetNewlyFlag() const NN_NOEXCEPT;

    /*!
        @brief      新規成立フラグを落とします。

        @return     処理結果。

        @details
                    新規成立フラグは、「マイページ」の一覧画面で新着マークを出すために利用します。@n
                    新規成立フラグは OFF から ON に変更することはできません。

                    新規成立フラグのみ、ネットワーク未接続時でも値を変更することが可能です。
    */
    nn::Result DropNewlyFlag() NN_NOEXCEPT;

    /*!
        @brief      お気に入りフラグを取得します。

        @return     お気に入りフラグ。
    */
    bool GetFavoriteFlag() const NN_NOEXCEPT;

    /*!
        @brief      お気に入りフラグを変更します。

        @param[out] outAsync    非同期コンテキスト。
        @param[in]  newValue    新しい設定値。

        @return     処理結果。

        @pre
            - outAsync != nullptr
            - NIFM ライブラリを初期化している。

        @details
                    本関数は通信処理を行うため、事前にインターネット利用要求を行い、それが受理されている必要があります。@n
                    通信処理のエラーハンドリングは、 @ref AsyncContext::GetResult を参照してください。
    */
    nn::Result ChangeFavoriteFlag(AsyncContext* outAsync, bool newValue) NN_NOEXCEPT;

    /*!
        @brief      オンライン通知フラグを取得します。

        @return     オンライン時オーバーレイ表示フラグ。

        @details
                    このフラグは、プレゼンスの状態がオフラインからオンラインに変化したタイミングでオーバーレイ表示するかどうかを決定します。@n
                    オンラインプレイ中からオンラインに変化しただけでは、オーバーレイ表示されません。
    */
    bool GetOnlineNotificationFlag() const NN_NOEXCEPT;

    /*!
        @brief      オンライン通知フラグを変更します。

        @param[out] outAsync    非同期コンテキスト。
        @param[in]  newValue    新しい設定値。

        @return     処理結果。

        @pre
            - outAsync != nullptr
            - NIFM ライブラリを初期化している。

        @details
                    本関数は通信処理を行うため、事前にインターネット利用要求を行い、それが受理されている必要があります。@n
                    通信処理のエラーハンドリングは、 @ref AsyncContext::GetResult を参照してください。
    */
    nn::Result ChangeOnlineNotificationFlag(AsyncContext* outAsync, bool newValue) NN_NOEXCEPT;

private:
    //
    NN_FRIENDS_CLASS_STORAGE(64);
};

}}
