﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/friends/friends_TypesAdmin.h>
#include <nn/friends/friends_Macros.h>

namespace nn { namespace friends {

/*!
    @brief      フレンド申請を扱うクラスです。

    @details
                あるユーザーアカウントに関するフレンド申請の情報を参照することができます。@n
                複数のユーザーアカウントが同一ユーザーから／へのフレンド申請を保持していたとしても、ユーザーアカウントごとに参照できる内容は異なります。
*/
class FriendRequest
{
public:
    /*!
        @brief      コンストラクタです。
    */
    FriendRequest() NN_NOEXCEPT;

    /*!
        @brief      フレンド申請 ID を取得します。

        @return     フレンド申請 ID。

        @details
                    対面フレンド申請の場合、本関数は無効な ID を返します。
    */
    RequestId GetRequestId() const NN_NOEXCEPT;

    /*!
        @brief      ネットワークサービスアカウント ID を取得します。

        @return     ネットワークサービスアカウント ID。
    */
    nn::account::NetworkServiceAccountId GetAccountId() const NN_NOEXCEPT;

    /*!
        @brief      ニックネームを取得します。

        @return     ニックネーム。
    */
    const nn::account::Nickname& GetNickname() const NN_NOEXCEPT;

    /*!
        @brief      ユーザーのプロフィール画像 URL を取得します。

        @param[out] outUrl      プロフィール画像 URL。
        @param[in]  imageSize   画像サイズ。

        @return     処理結果。

        @pre
            - outUrl != nullptr

        @details
                    対面フレンド申請の場合、 @ref GetFacedFriendRequestProfileImage に指定するパスが記述されます。
    */
    nn::Result GetProfileImageUrl(Url* outUrl, ImageSize imageSize) const NN_NOEXCEPT;

    /*!
        @brief      フレンド申請手段を取得します。

        @return     フレンド申請手段。
    */
    RequestType GetRequestType() const NN_NOEXCEPT;

    /*!
        @brief      フレンド申請の状態を取得します。

        @return     フレンド申請の状態。
    */
    RequestStatus GetRequestStatus() const NN_NOEXCEPT;

    /*!
        @brief      フレンド申請の経路情報を取得します。

        @return     経路情報。

        @details
                    経路情報には、以下が含まれます。

                    - フレンド申請を行ったアプリケーションの情報@n
                      フレンド申請手段が @ref RequestType_InApp の場合のみ、有効な値が入ります。
                    - フレンド申請時の相手のアプリ内表示名@n
                      フレンド申請手段が @ref RequestType_InApp 、または、 @ref RequestType_FriendCandidate の場合、有効な値が入ります。
                    - フレンド申請した時刻
    */
    const RouteInfo& GetRouteInfo() const NN_NOEXCEPT;

    /*!
        @brief      外部アプリケーションのカタログ ID を取得します。（フレンド候補専用）

        @return     外部アプリケーションのカタログ ID。

        @details
                    本関数は、フレンド申請手段が @ref RequestType_FriendCandidate の場合のみ、有効な値が入ります。

                    本関数で取得したカタログ ID を使用することで、外部アプリケーションのカタログを取得することができます。

        @see
            - GetExternalApplicationCatalog
    */
    const ExternalApplicationCatalogId& GetExternalApplicationCatalogId() const NN_NOEXCEPT;

    /*!
        @brief      Mii の名前を取得します。（NNID フレンド専用）

        @return     Mii の名前。

        @details
                    本関数は、フレンド申請手段が @ref RequestType_3Ds 、または、 @ref RequestType_Nnid の場合のみ、有効な値が入ります。
    */
    const MiiName& GetMiiName() const NN_NOEXCEPT;

    /*!
        @brief      Mii 画像 URL パラメータを取得します。（NNID フレンド専用）

        @return     Mii 画像 URL パラメータ。

        @details
                    本関数は、フレンド申請手段が @ref RequestType_3Ds 、または、 @ref RequestType_Nnid の場合のみ、有効な値が入ります。
    */
    const MiiImageUrlParam& GetMiiImageUrlParam() const NN_NOEXCEPT;

    /*!
        @brief      Mii 画像 URL を取得します。（NNID フレンド専用）

        @param[out] outUrl  Mii 画像 URL。

        @return     処理結果。

        @pre
            - outUrl != nullptr

        @details
                    本関数は、フレンド申請手段が @ref RequestType_3Ds 、または、 @ref RequestType_Nnid の場合のみ、有効な値が入ります。
    */
    nn::Result GetMiiImageUrl(Url* outUrl) const NN_NOEXCEPT;

    /*!
        @brief      既読かどうかを確認します。

        @return     既読かどうか。

        @details
                    本関数は、本オブジェクトが受信側のフレンド申請である場合のみ、有効な値を返します。@n
                    本オブジェクトが送信側のフレンド申請だった場合、常に true を返します。
    */
    bool IsRead() const NN_NOEXCEPT;

    /*!
        @brief      有効なデータを保持しているかどうかを確認します。

        @return     有効なデータを保持しているかどうか。
    */
    bool IsValid() const NN_NOEXCEPT;

private:
    //
    NN_FRIENDS_CLASS_STORAGE(512);
};

}}
