﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/friends/friends_TypesAdmin.h>
#include <nn/friends/friends_ResultAdmin.h>
#include <nn/friends/friends_ApiFriendRequest.h>
#include <nn/friends/friends_DaemonSuspension.h>
#include <nn/friends/friends_FriendDetailedInfo.h>
#include <nn/friends/friends_FriendRequest.h>
#include <nn/friends/friends_FriendCandidate.h>
#include <nn/friends/friends_NintendoNetworkIdFriend.h>
#include <nn/friends/friends_BlockedUser.h>
#include <nn/friends/friends_PlayHistory.h>
#include <nn/friends/friends_PresenceView.h>
#include <nn/friends/friends_ProfileExtra.h>
#include <nn/friends/friends_SnsAccountFriend.h>
#include <nn/friends/friends_AsyncContext.h>
#include <nn/friends/friends_Option.h>
#include <nn/friends/friends_OptionAdmin.h>

namespace nn { namespace friends {

/*!
    @name ライブラリ設定
    @{
*/

/*!
    @brief      ライブラリオプションを設定します。

    @param[in]  option  オプション。
    @param[in]  ...     オプション値。

    @details
                オプションによって指定する値は異なります。@n
                詳細は、 @ref OptionAdmin を参照してください。
*/
void SetOption(int32_t option, ...) NN_NOEXCEPT;

/*!
    @}
    @name フレンド
    @{
*/

/*!
    @brief      フレンドの人数を取得します。

    @param[out] outCount    条件に合致したフレンドの人数。
    @param[in]  uid         ユーザーアカウント。
    @param[in]  filter      フレンドリスト取得のフィルター。

    @return     処理結果。

    @pre
        - outCount != nullptr
        - 指定したユーザーアカウントが Open 状態である。

    @details
                本体ストレージ上のキャッシュから条件に合致したフレンドの人数を取得します。
*/
nn::Result GetFriendCount(int* outCount,
    const nn::account::Uid& uid, const FriendFilter& filter) NN_NOEXCEPT;

/*!
    @brief      新着フレンドの人数を取得します。

    @param[out] outCount    新着フレンドの人数。
    @param[in]  uid         ユーザーアカウント。

    @return     処理結果。

    @pre
        - outCount != nullptr
        - 指定したユーザーアカウントが Open 状態である。

    @details
                本体ストレージ上のキャッシュから新着フレンドの人数を取得します。
*/
nn::Result GetNewlyFriendCount(int* outCount,
    const nn::account::Uid& uid) NN_NOEXCEPT;

/*!
    @brief      フレンドの詳細情報を取得します。

    @param[out] outInfo     フレンドの詳細情報。
    @param[in]  uid         ユーザーアカウント。
    @param[in]  accountId   フレンドのネットワークサービスアカウント ID。

    @return     処理結果。

    @pre
        - outInfo != nullptr
        - 指定したユーザーアカウントが Open 状態である。

    @details
                本関数は本体ストレージ上のキャッシュを取得します。@n
                そのため、ネットワーク未接続時でも取得することができます。

                フレンドサーバーと通信可能な状態であれば、本体ストレージ上のキャッシュは定期的に更新されます。
*/
nn::Result GetFriendDetailedInfo(FriendDetailedInfo* outInfo,
    const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT;

/*!
    @brief      フレンドの新規成立フラグをすべて落とします。

    @param[in]  uid ユーザーアカウント。

    @return     処理結果。

    @pre
        - 指定したユーザーアカウントが Open 状態である。

    @details
                本関数は、指定したユーザーアカウントの全フレンドに対して @ref FriendSetting::DropNewlyFlag を一括で行います。@n
                詳しくは、 @ref FriendSetting::DropNewlyFlag を参照してください。

                新規成立フラグのみ、ネットワーク未接続時でも値を変更することが可能です。
*/
nn::Result DropFriendNewlyFlags(const nn::account::Uid& uid) NN_NOEXCEPT;

/*!
    @brief      フレンドを削除します。

    @param[out] outAsync    非同期コンテキスト。
    @param[in]  uid         ユーザーアカウント。
    @param[in]  accountId   フレンドのネットワークサービスアカウント ID。

    @return     処理結果。

    @pre
        - outAsync != nullptr
        - 指定したユーザーアカウントが Open 状態である。
        - NIFM ライブラリを初期化している。

    @details
                本関数は通信処理を行うため、事前にインターネット利用要求を行い、それが受理されている必要があります。@n
                通信処理のエラーハンドリングは、 @ref AsyncContext::GetResult を参照してください。
*/
nn::Result DeleteFriend(AsyncContext* outAsync,
    const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT;

/*!
    @brief      本体ストレージ上のフレンドリストのキャッシュを最新の状態に更新します。

    @param[out] outAsync    非同期コンテキスト。
    @param[in]  uid         ユーザーアカウント。

    @return     処理結果。

    @pre
        - outAsync != nullptr
        - 指定したユーザーアカウントが Open 状態である。
        - NIFM ライブラリを初期化している。

    @details
                本関数は、任意のタイミングでサーバーと同期処理を行います。@n
                通常、ユーザーアカウント選択時やフレンドプレゼンスの更新通知時にバックグラウンドで同期処理が行われます。

                通信処理のエラーハンドリングは、 @ref AsyncContext::GetResult を参照してください。

    @see
        - GetFriendList
*/
nn::Result SyncFriendList(AsyncContext* outAsync,
    const nn::account::Uid& uid) NN_NOEXCEPT;

/*!
    @brief      本体ストレージ上のフレンドリストのキャッシュを最新の状態に更新するバックグラウンドタスクの実行を要求します。

    @param[in]  uid ユーザーアカウント。

    @return     処理結果。

    @pre
        - 指定したユーザーアカウントが Open 状態である。

    @details
                本関数は、バックグラウンドで行われるフレンドリストの同期処理を明示的に要求します。@n
                @ref SyncFriendList と違い、同期処理の開始タイミングや完了タイミングを検知することはできません。

    @see
        - GetFriendList
*/
nn::Result RequestSyncFriendList(const nn::account::Uid& uid) NN_NOEXCEPT;

/*!
    @}
    @name フレンド申請
    @{
*/

/*!
    @brief      受信したフレンド申請件数を取得します。

    @param[out] outUnreadCount  未読件数。
    @param[out] outReadCount    既読件数。
    @param[in]  uid             ユーザーアカウント。

    @return     処理結果。

    @pre
        - outUnreadCount != nullptr
        - outReadCount != nullptr
        - 指定したユーザーアカウントが Open 状態である。

    @details
                本関数は、システムがキャッシュした最新の件数を取得します。@n
                このキャッシュは、スリープ復帰時や受信側のフレンド申請取得時等に更新されます。
*/
nn::Result GetReceivedFriendRequestCount(int* outUnreadCount, int* outReadCount,
    const nn::account::Uid& uid) NN_NOEXCEPT;

/*!
    @brief      フレンド申請リストを取得します。

    @param[out] outAsync    非同期コンテキスト。
    @param[out] outCount    取得した数。
    @param[out] outRequests フレンド申請リスト。
    @param[in]  uid         ユーザーアカウント。
    @param[in]  offset      オフセット。
    @param[in]  count       フレンド申請リストの要素数。
    @param[in]  listType    フレンド申請リスト種別。

    @return     処理結果。

    @pre
        - outAsync != nullptr
        - outCount != nullptr
        - outRequests != nullptr
        - offset >= 0
        - count > 0
        - 指定したユーザーアカウントが Open 状態である。
        - NIFM ライブラリを初期化している。

    @details
                指定したオフセットから指定した数のフレンド申請情報を取得します。

                受信側のフレンド申請の件数は、仕様上無制限です。@n
                受信側のフレンド申請を全件取得したい場合、取得した数が 0 になるまでオフセットをずらしならが繰り返し取得してください。

                本関数は通信処理を行うため、事前にインターネット利用要求を行い、それが受理されている必要があります。@n
                通信処理のエラーハンドリングは、 @ref AsyncContext::GetResult を参照してください。
*/
nn::Result GetFriendRequestList(AsyncContext* outAsync, int* outCount, FriendRequest* outRequests,
    const nn::account::Uid& uid, int offset, int count, RequestListType listType) NN_NOEXCEPT;

/*!
    @brief      フレンド申請を行います。

    @param[out] outAsync    非同期コンテキスト。
    @param[in]  uid         ユーザーアカウント。
    @param[in]  accountId   相手のネットワークサービスアカウント ID。
    @param[in]  requestType フレンド申請手段。

    @return     処理結果。

    @pre
        - outAsync != nullptr
        - 指定したユーザーアカウントが Open 状態である。
        - NIFM ライブラリを初期化している。

    @details
                本関数は通信処理を行うため、事前にインターネット利用要求を行い、それが受理されている必要があります。@n
                通信処理のエラーハンドリングは、 @ref AsyncContext::GetResult を参照してください。

                非同期処理の結果が以下のいずれかの場合、ユーザーにエラー原因の解消を促すことを推奨します。

                - ResultMyFriendListLimitReached@n
                  フレンドと送信したフレンド申請の数の合計が、 @ref FriendCountMax に達している。
                - ResultBlockedByMe@n
                  相手をブロックしている。
*/
nn::Result SendFriendRequest(AsyncContext* outAsync,
    const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId, RequestType requestType) NN_NOEXCEPT;

/*!
    @brief      いっしょに遊んだ記録付きでフレンド申請を行います。

    @param[out] outAsync            非同期コンテキスト。
    @param[in]  uid                 ユーザーアカウント。
    @param[in]  accountId           相手のネットワークサービスアカウント ID。
    @param[in]  requestType         フレンド申請手段。
    @param[in]  appInfo             いっしょに遊んだアプリケーション。
    @param[in]  inAppScreenName     相手のアプリ内表示名。
    @param[in]  myInAppScreenName   自分のアプリ内表示名。

    @return     処理結果。

    @pre
        - outAsync != nullptr
        - 指定したユーザーアカウントが Open 状態である。
        - NIFM ライブラリを初期化している。

    @details
                本関数は通信処理を行うため、事前にインターネット利用要求を行い、それが受理されている必要があります。@n
                通信処理のエラーハンドリングは、 @ref AsyncContext::GetResult を参照してください。

                非同期処理の結果が以下のいずれかの場合、ユーザーにエラー原因の解消を促すことを推奨します。

                - ResultMyFriendListLimitReached@n
                  フレンドと送信したフレンド申請の数の合計が、 @ref FriendCountMax に達している。
                - ResultBlockedByMe@n
                  相手をブロックしている。
*/
nn::Result SendFriendRequest(AsyncContext* outAsync,
    const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId, RequestType requestType,
    const ApplicationInfo& appInfo, const InAppScreenName& inAppScreenName, const InAppScreenName& myInAppScreenName) NN_NOEXCEPT;

/*!
    @brief      いっしょに遊んだ記録付きでフレンド申請を行います。（フレンド候補用）

    @param[out] outAsync            非同期コンテキスト。
    @param[in]  uid                 ユーザーアカウント。
    @param[in]  accountId           相手のネットワークサービスアカウント ID。
    @param[in]  requestType         フレンド申請手段。
    @param[in]  catalogId           いっしょに遊んだ外部アプリケーションのカタログ ID。
    @param[in]  inAppScreenName     相手のアプリ内表示名。
    @param[in]  myInAppScreenName   自分のアプリ内表示名。

    @return     処理結果。

    @pre
        - outAsync != nullptr
        - 指定したユーザーアカウントが Open 状態である。
        - NIFM ライブラリを初期化している。

    @details
                本関数は通信処理を行うため、事前にインターネット利用要求を行い、それが受理されている必要があります。@n
                通信処理のエラーハンドリングは、 @ref AsyncContext::GetResult を参照してください。

                非同期処理の結果が以下のいずれかの場合、ユーザーにエラー原因の解消を促すことを推奨します。

                - ResultMyFriendListLimitReached@n
                  フレンドと送信したフレンド申請の数の合計が、 @ref FriendCountMax に達している。
                - ResultBlockedByMe@n
                  相手をブロックしている。
*/
nn::Result SendFriendRequest(AsyncContext* outAsync,
    const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId, RequestType requestType,
    const ExternalApplicationCatalogId& catalogId, const InAppScreenName& inAppScreenName, const InAppScreenName& myInAppScreenName) NN_NOEXCEPT;

/*!
    @brief      フレンド申請を行います。（NNID フレンド用）

    @param[out] outAsync            非同期コンテキスト。
    @param[in]  uid                 ユーザーアカウント。
    @param[in]  accountId           相手のネットワークサービスアカウント ID。
    @param[in]  requestType         フレンド申請手段。
    @param[in]  miiName             相手の Mii の名前。
    @param[in]  miiImageUrlParam    相手の Mii 画像 URL パラメータ。
    @param[in]  myMiiName           自分の Mii の名前。
    @param[in]  myMiiImageUrlParam  自分の Mii 画像 URL パラメータ。

    @return     処理結果。

    @pre
        - outAsync != nullptr
        - 指定したユーザーアカウントが Open 状態である。
        - NIFM ライブラリを初期化している。

    @details
                本関数は通信処理を行うため、事前にインターネット利用要求を行い、それが受理されている必要があります。@n
                通信処理のエラーハンドリングは、 @ref AsyncContext::GetResult を参照してください。

                非同期処理の結果が以下のいずれかの場合、ユーザーにエラー原因の解消を促すことを推奨します。

                - ResultMyFriendListLimitReached@n
                  フレンドと送信したフレンド申請の数の合計が、 @ref FriendCountMax に達している。
                - ResultBlockedByMe@n
                  相手をブロックしている。
*/
nn::Result SendFriendRequest(AsyncContext* outAsync,
    const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId, RequestType requestType,
    const MiiName& miiName, const MiiImageUrlParam& miiImageUrlParam, const MiiName& myMiiName, const MiiImageUrlParam& myMiiImageUrlParam) NN_NOEXCEPT;

/*!
    @brief      フレンド申請を取り下げます。

    @param[out] outAsync    非同期コンテキスト。
    @param[in]  uid         ユーザーアカウント。
    @param[in]  requestId   フレンド申請 ID。

    @return     処理結果。

    @pre
        - outAsync != nullptr
        - 指定したユーザーアカウントが Open 状態である。
        - NIFM ライブラリを初期化している。

    @details
                本関数は通信処理を行うため、事前にインターネット利用要求を行い、それが受理されている必要があります。@n
                通信処理のエラーハンドリングは、 @ref AsyncContext::GetResult を参照してください。

                非同期処理の結果が ResultRequestStateAlreadyChanged の場合、以下のいずれかが原因となります。

                - 相手がフレンド申請を承認、または、拒否した。
                - フレンド申請をすでに取り下げている。
                - フレンド申請の有効期間が過ぎた。

                この場合、フレンド申請リストの再取得を行うことを推奨します。

                対面フレンド申請を取り下げる場合、 @ref CancelFacedFriendRequest を使用してください。
*/
nn::Result CancelFriendRequest(AsyncContext* outAsync,
    const nn::account::Uid& uid, RequestId requestId) NN_NOEXCEPT;

/*!
    @brief      フレンド申請を承認します。

    @param[out] outAsync    非同期コンテキスト。
    @param[in]  uid         ユーザーアカウント。
    @param[in]  requestId   フレンド申請 ID。

    @return     処理結果。

    @pre
        - outAsync != nullptr
        - 指定したユーザーアカウントが Open 状態である。
        - NIFM ライブラリを初期化している。

    @details
                本関数は通信処理を行うため、事前にインターネット利用要求を行い、それが受理されている必要があります。@n
                通信処理のエラーハンドリングは、 @ref AsyncContext::GetResult を参照してください。

                非同期処理の結果が ResultRequestStateAlreadyChanged の場合、以下のいずれかが原因となります。

                - 相手がフレンド申請を取り下げた。
                - フレンド申請をすでに承認、または、拒否している。
                - フレンド申請の有効期間が過ぎた。

                この場合、フレンド申請リストの再取得を行うことを推奨します。
*/
nn::Result AcceptFriendRequest(AsyncContext* outAsync,
    const nn::account::Uid& uid, RequestId requestId) NN_NOEXCEPT;

/*!
    @brief      フレンド申請を拒否します。

    @param[out] outAsync    非同期コンテキスト。
    @param[in]  uid         ユーザーアカウント。
    @param[in]  requestId   フレンド申請 ID。

    @return     処理結果。

    @pre
        - outAsync != nullptr
        - 指定したユーザーアカウントが Open 状態である。
        - NIFM ライブラリを初期化している。

    @details
                本関数は通信処理を行うため、事前にインターネット利用要求を行い、それが受理されている必要があります。@n
                通信処理のエラーハンドリングは、 @ref AsyncContext::GetResult を参照してください。

                非同期処理の結果が ResultRequestStateAlreadyChanged の場合、以下のいずれかが原因となります。

                - 相手がフレンド申請を取り下げた。
                - フレンド申請をすでに承認、または、拒否している。
                - フレンド申請の有効期間が過ぎた。

                この場合、フレンド申請リストの再取得を行うことを推奨します。
*/
nn::Result RejectFriendRequest(AsyncContext* outAsync,
    const nn::account::Uid& uid, RequestId requestId) NN_NOEXCEPT;

/*!
    @brief      フレンド申請を既読にします。

    @param[out] outAsync    非同期コンテキスト。
    @param[in]  uid         ユーザーアカウント。
    @param[in]  requestId   フレンド申請 ID。

    @return     処理結果。

    @pre
        - outAsync != nullptr
        - 指定したユーザーアカウントが Open 状態である。
        - NIFM ライブラリを初期化している。

    @details
                本関数は通信処理を行うため、事前にインターネット利用要求を行い、それが受理されている必要があります。@n
                通信処理のエラーハンドリングは、 @ref AsyncContext::GetResult を参照してください。

                非同期処理の結果が ResultRequestStateAlreadyChanged の場合、以下のいずれかが原因となります。

                - 相手がフレンド申請を取り下げた。
                - フレンド申請をすでに承認、または、拒否している。
                - フレンド申請の有効期間が過ぎた。

                この場合、フレンド申請リストの再取得を行うことを推奨します。
*/
nn::Result ReadFriendRequest(AsyncContext* outAsync,
    const nn::account::Uid& uid, RequestId requestId) NN_NOEXCEPT;

/*!
    @brief      対面フレンド申請を追加します。

    @param[in]  uid         ユーザーアカウント。
    @param[in]  key         相手の登録キー。
    @param[in]  nickname    相手のニックネーム。
    @param[in]  image       相手のプロフィール画像。
    @param[in]  imageSize   相手のプロフィール画像のサイズ。

    @return     処理結果。

    @pre
        - image != nullptr
        - 0 < imageSize <= ProfileImageSizeMax
        - 指定したユーザーアカウントが Open 状態である。

    @details
                ニックネームとプロフィール画像は、必ず @ref nn::account::GetNickname と @ref nn::account::LoadProfileImage で取得したものを指定してください。

                対面フレンド申請は本体ストレージ上に保存され、ネットワーク接続時に自動送信されます。@n
                送信に成功した、または、相手がすでにフレンドだった場合、対面フレンド申請は自動送信後に削除されます。

    @see
        - nn::account::GetNickname
        - nn::account::LoadProfileImage
*/
nn::Result AddFacedFriendRequest(const nn::account::Uid& uid,
    const FacedFriendRequestRegistrationKey& key,
    const nn::account::Nickname& nickname, const void* image, size_t imageSize) NN_NOEXCEPT;

/*!
    @brief      対面フレンド申請を取り下げます。

    @param[in]  uid         ユーザーアカウント。
    @param[in]  accountId   相手のネットワークサービスアカウント ID。

    @return     処理結果。

    @pre
        - 指定したユーザーアカウントが Open 状態である。

    @details
                本関数はインフラ未接続時でも利用することができます。
*/
nn::Result CancelFacedFriendRequest(const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT;

/*!
    @brief      対面フレンド申請のプロフィール画像を取得します。

    @param[out] outSize     取得したサイズ。
    @param[in]  uid         ユーザーアカウント。
    @param[in]  accountId   相手のネットワークサービスアカウント ID。
    @param[in]  buffer      バッファ。
    @param[in]  size        バッファのサイズ。

    @return     処理結果。

    @pre
        - outSize != nullptr
        - buffer != nullptr
        - size > 0
        - 指定したユーザーアカウントが Open 状態である。
*/
nn::Result GetFacedFriendRequestProfileImage(size_t* outSize,
    const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId, void* buffer, size_t size) NN_NOEXCEPT;

/*!
    @brief      対面フレンド申請のプロフィール画像を取得します。

    @param[out] outSize     取得したサイズ。
    @param[in]  path        画像パス。
    @param[in]  buffer      バッファ。
    @param[in]  size        バッファのサイズ。

    @return     処理結果。

    @pre
        - outSize != nullptr
        - path != nullptr
        - buffer != nullptr
        - size > 0

    @details
                @ref FriendRequest::GetProfileImageUrl で取得した URL を画像パスに指定することで、プロフィール画像を取得できます。
*/
nn::Result GetFacedFriendRequestProfileImage(size_t* outSize,
    const char* path, void* buffer, size_t size) NN_NOEXCEPT;

/*!
    @brief      対面フレンド申請の再送を行います。

    @param[out] outAsync    非同期コンテキスト。
    @param[in]  uid         ユーザーアカウント。
    @param[in]  accountId   相手のネットワークサービスアカウント ID。

    @return     処理結果。

    @pre
        - outAsync != nullptr
        - 指定したユーザーアカウントが Open 状態である。
        - NIFM ライブラリを初期化している。

    @details
                @ref AddFacedFriendRequest で追加した対面フレンド申請の再送を行います。

                本関数は通信処理を行うため、事前にインターネット利用要求を行い、それが受理されている必要があります。@n
                通信処理のエラーハンドリングは、 @ref AsyncContext::GetResult を参照してください。

                非同期処理の結果が以下のいずれかの場合、ユーザーにエラー原因の解消を促すことを推奨します。

                - ResultMyFriendListLimitReached@n
                  フレンドと送信したフレンド申請の数の合計が、 @ref FriendCountMax に達している。
                - ResultBlockedByMe@n
                  相手をブロックしている。

                非同期処理の結果が以下のいずれかの場合、対面フレンド申請は自動的に削除されます。

                - ResultNetworkServiceAccountNotExists
                - ResultAlreadyRequested
                - ResultAlreadyFriend
*/
nn::Result ResendFacedFriendRequest(AsyncContext* outAsync,
    const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT;

/*!
    @}
    @name フレンド候補
    @{
*/

/*!
    @brief      フレンド候補リストを取得します。

    @param[out] outAsync        非同期コンテキスト。
    @param[out] outCount        取得した数。
    @param[out] outCandidates   フレンド候補リスト。
    @param[in]  uid             ユーザーアカウント。
    @param[in]  count           フレンド候補リストの要素数。
    @param[in]  type            フレンド候補の種類。

    @return     処理結果。

    @pre
        - outAsync != nullptr
        - outCount != nullptr
        - outCandidates != nullptr
        - count > 0
        - 指定したユーザーアカウントが Open 状態である。
        - NIFM ライブラリを初期化している。

    @details
                本関数は通信処理を行うため、事前にインターネット利用要求を行い、それが受理されている必要があります。@n
                通信処理のエラーハンドリングは、 @ref AsyncContext::GetResult を参照してください。
*/
nn::Result GetFriendCandidateList(AsyncContext* outAsync, int* outCount, FriendCandidate* outCandidates,
    const nn::account::Uid& uid, int count, FriendCandidateType type) NN_NOEXCEPT;

/*!
    @brief      NNID に紐付く自身の情報とフレンドリストを取得します。

    @param[out] outAsync    非同期コンテキスト。
    @param[out] outUserInfo NNID の自身のユーザー情報。
    @param[out] outCount    取得した数。
    @param[out] outFriends  NNID のフレンドリスト。
    @param[in]  uid         ユーザーアカウント。
    @param[in]  count       NNID のフレンドリストの要素数。
    @param[in]  type        NNID アカウントの種類。

    @return     処理結果。

    @pre
        - outAsync != nullptr
        - outUserInfo != nullptr
        - outCount != nullptr
        - outFriends != nullptr
        - count > 0
        - 指定したユーザーアカウントが Open 状態である。
        - NIFM ライブラリを初期化している。

    @details
                本関数で取得できるフレンドリストは、NNID をニンテンドーアカウントに紐付けているフレンドに限定されます。@n
                紐付けの有無に関わらずフレンドの人数を知りたい場合、 NintendoNetworkIdUserInfo::friendCount を参照してください。

                本関数は通信処理を行うため、事前にインターネット利用要求を行い、それが受理されている必要があります。@n
                通信処理のエラーハンドリングは、 @ref AsyncContext::GetResult を参照してください。
*/
nn::Result GetNintendoNetworkIdInfo(AsyncContext* outAsync, NintendoNetworkIdUserInfo* outUserInfo,
    int* outCount, NintendoNetworkIdFriend* outFriends,
    const nn::account::Uid& uid, int count, NintendoNetworkIdAccountType type) NN_NOEXCEPT;

/*!
    @brief      フレンド候補取得用 SNS アカウントをニンテンドーアカウントに紐付けるための WEB ページ URL を取得します。

    @param[out] outAsync    非同期コンテキスト。
    @param[out] outUrl      URL。
    @param[in]  uid         ユーザーアカウント。
    @param[in]  type        SNS アカウントの種類。

    @return     処理結果。

    @pre
        - outAsync != nullptr
        - outUrl != nullptr
        - NIFM ライブラリを初期化している。

    @details
                本関数で取得した URL は、ブラウザに渡してください。@n
                ユーザーがブラウザ上で紐付け手続きを完了させることで、フレンド候補取得用 SNS アカウントとニンテンドーアカウントが紐付きます。

                本関数は通信処理を行うため、事前にインターネット利用要求を行い、それが受理されている必要があります。@n
                通信処理のエラーハンドリングは、 @ref AsyncContext::GetResult を参照してください。
*/
nn::Result GetSnsAccountLinkPageUrl(AsyncContext* outAsync, WebPageUrl* outUrl,
    const nn::account::Uid& uid, SnsAccountType type) NN_NOEXCEPT;

/*!
    @brief      フレンド候補取得用 SNS アカウントとニンテンドーアカウントの紐付けを解除します。

    @param[out] outAsync    非同期コンテキスト。
    @param[in]  uid         ユーザーアカウント。
    @param[in]  type        SNS アカウントの種類。

    @return     処理結果。

    @pre
        - outAsync != nullptr
        - NIFM ライブラリを初期化している。

    @details
                SNS アカウントが紐付いていない状態で本関数を呼び出しても、通信環境やニンテンドーアカウントの状態に問題がなければ非同期処理は成功します。

                本関数は通信処理を行うため、事前にインターネット利用要求を行い、それが受理されている必要があります。@n
                通信処理のエラーハンドリングは、 @ref AsyncContext::GetResult を参照してください。
*/
nn::Result UnlinkSnsAccount(AsyncContext* outAsync,
    const nn::account::Uid& uid, SnsAccountType type) NN_NOEXCEPT;

/*!
    @brief      フレンド候補取得用 SNS アカウントの紐付け状態を取得します。

    @param[out] outAsync    非同期コンテキスト。
    @param[out] outLinkage  SNS アカウントの紐付け状態。
    @param[in]  uid         ユーザーアカウント。

    @return     処理結果。

    @pre
        - outAsync != nullptr
        - outLinkage != nullptr
        - NIFM ライブラリを初期化している。

    @details
                本関数は通信処理を行うため、事前にインターネット利用要求を行い、それが受理されている必要があります。@n
                通信処理のエラーハンドリングは、 @ref AsyncContext::GetResult を参照してください。
*/
nn::Result GetSnsAccountLinkage(AsyncContext* outAsync, SnsAccountLinkage* outLinkage,
    const nn::account::Uid& uid) NN_NOEXCEPT;

/*!
    @brief      フレンド候補取得用 SNS アカウントのプロフィールを取得します。

    @param[out] outAsync    非同期コンテキスト。
    @param[out] outProfile  SNS アカウントのプロフィール。
    @param[in]  uid         ユーザーアカウント。
    @param[in]  accountId   ネットワークサービスアカウント ID。
    @param[in]  type        SNS アカウントの種類。

    @return     処理結果。

    @pre
        - outAsync != nullptr
        - outProfile != nullptr
        - NIFM ライブラリを初期化している。

    @details
                本関数は、自分のフレンド候補取得用 SNS アカウントの権限でプロフィールの取得を行います。@n
                自分以外のネットワークサービスアカウント ID を指定した場合、本関数の呼び出し時点で SNS 上でフレンド関係が解消されていると取得に失敗します。

                本関数は通信処理を行うため、事前にインターネット利用要求を行い、それが受理されている必要があります。@n
                通信処理のエラーハンドリングは、 @ref AsyncContext::GetResult を参照してください。
*/
nn::Result GetSnsAccountProfile(AsyncContext* outAsync, SnsAccountProfile* outProfile,
    const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId, SnsAccountType type) NN_NOEXCEPT;

/*!
    @brief      フレンド候補取得用 SNS アカウントのフレンドリストを取得します。

    @param[out] outAsync    非同期コンテキスト。
    @param[out] outCount    取得した数。
    @param[out] outFriends  SNS アカウントのフレンドリスト。
    @param[in]  uid         ユーザーアカウント。
    @param[in]  count       SNS アカウントのフレンドリストの要素数。
    @param[in]  type        SNS アカウントの種類。

    @return     処理結果。

    @pre
        - outAsync != nullptr
        - outCount != nullptr
        - outFriends != nullptr
        - count > 0
        - NIFM ライブラリを初期化している。

    @details
                本関数で取得できるフレンドリストは、フレンド候補取得用の SNS アカウントをニンテンドーアカウントに紐付けているフレンドに限定されます。

                本関数は通信処理を行うため、事前にインターネット利用要求を行い、それが受理されている必要があります。@n
                通信処理のエラーハンドリングは、 @ref AsyncContext::GetResult を参照してください。
*/
nn::Result GetSnsAccountFriendList(AsyncContext* outAsync, int* outCount, SnsAccountFriend* outFriends,
    const nn::account::Uid& uid, int count, SnsAccountType type) NN_NOEXCEPT;

/*!
    @}
    @name ブロック
    @{
*/

/*!
    @brief      ブロックリストを取得します。

    @param[out] outCount    取得した数。
    @param[out] outUsers    ブロックリスト。
    @param[in]  uid         ユーザーアカウント。
    @param[in]  offset      オフセット。
    @param[in]  count       ブロックリストの要素数。

    @return     処理結果。

    @pre
        - outCount != nullptr
        - outUsers != nullptr
        - offset >= 0
        - count > 0
        - 指定したユーザーアカウントが Open 状態である。

    @details
                指定したオフセットから指定した数のブロックしたユーザー情報を取得します。

                本関数は本体ストレージ上のキャッシュからブロックリストを取得します。@n
                そのため、ネットワーク未接続時でもブロックリストを取得することができます。

                フレンドサーバーと通信可能な状態であれば、本体ストレージ上のキャッシュは定期的に更新されます。
*/
nn::Result GetBlockedUserList(int* outCount, BlockedUser* outUsers,
    const nn::account::Uid& uid, int offset, int count) NN_NOEXCEPT;

/*!
    @brief      ブロックリストにユーザーを登録します。

    @param[out] outAsync    非同期コンテキスト。
    @param[in]  uid         ユーザーアカウント。
    @param[in]  accountId   相手のネットワークサービスアカウント ID。
    @param[in]  blockReason ブロック理由。

    @return     処理結果。

    @pre
        - outAsync != nullptr
        - 指定したユーザーアカウントが Open 状態である。
        - NIFM ライブラリを初期化している。

    @details
                本関数は通信処理を行うため、事前にインターネット利用要求を行い、それが受理されている必要があります。@n
                通信処理のエラーハンドリングは、 @ref AsyncContext::GetResult を参照してください。

                非同期処理の結果が以下のいずれかの場合、ユーザーにエラー原因の解消を促すことを推奨します。

                - ResultAlreadyBlocked@n
                  すでに相手をブロックしている。
                - ResultBlockListLimitReached@n
                  ブロックしたユーザー数が、 @ref BlockedUserCountMax に達している。
*/
nn::Result BlockUser(AsyncContext* outAsync,
    const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId, BlockReason blockReason) NN_NOEXCEPT;

/*!
    @brief      いっしょに遊んだ記録付きでブロックリストにユーザーを登録します。

    @param[out] outAsync        非同期コンテキスト。
    @param[in]  uid             ユーザーアカウント。
    @param[in]  accountId       相手のネットワークサービスアカウント ID。
    @param[in]  blockReason     ブロック理由。
    @param[in]  appInfo         いっしょに遊んだアプリケーション。
    @param[in]  inAppScreenName 相手のアプリ内表示名。

    @return     処理結果。

    @pre
        - outAsync != nullptr
        - 指定したユーザーアカウントが Open 状態である。
        - NIFM ライブラリを初期化している。

    @details
                本関数は通信処理を行うため、事前にインターネット利用要求を行い、それが受理されている必要があります。@n
                通信処理のエラーハンドリングは、 @ref AsyncContext::GetResult を参照してください。

                非同期処理の結果が以下のいずれかの場合、ユーザーにエラー原因の解消を促すことを推奨します。

                - ResultAlreadyBlocked@n
                  すでに相手をブロックしている。
                - ResultBlockListLimitReached@n
                  ブロックしたユーザー数が、 @ref BlockedUserCountMax に達している。
*/
nn::Result BlockUser(AsyncContext* outAsync,
    const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId, BlockReason blockReason,
    const ApplicationInfo& appInfo, const InAppScreenName& inAppScreenName) NN_NOEXCEPT;

/*!
    @brief      ブロックを解除します。

    @param[out] outAsync    非同期コンテキスト。
    @param[in]  uid         ユーザーアカウント。
    @param[in]  accountId   相手のネットワークサービスアカウント ID。

    @return     処理結果。

    @pre
        - outAsync != nullptr
        - 指定したユーザーアカウントが Open 状態である。
        - NIFM ライブラリを初期化している。

    @details
                本関数は通信処理を行うため、事前にインターネット利用要求を行い、それが受理されている必要があります。@n
                通信処理のエラーハンドリングは、 @ref AsyncContext::GetResult を参照してください。
*/
nn::Result UnblockUser(AsyncContext* outAsync,
    const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT;

/*!
    @brief      本体ストレージ上のブロックリストのキャッシュを最新の状態に更新します。

    @param[out] outAsync    非同期コンテキスト。
    @param[in]  uid         ユーザーアカウント。

    @return     処理結果。

    @pre
        - outAsync != nullptr
        - 指定したユーザーアカウントが Open 状態である。
        - NIFM ライブラリを初期化している。

    @details
                本関数は、任意のタイミングでサーバーと同期処理を行います。@n
                通常、ユーザーアカウント選択時にバックグラウンドで同期処理が行われます。

                通信処理のエラーハンドリングは、 @ref AsyncContext::GetResult を参照してください。

    @see
        - GetBlockedUserList
*/
nn::Result SyncBlockedUserList(AsyncContext* outAsync,
    const nn::account::Uid& uid) NN_NOEXCEPT;

/*!
    @}
    @name プロフィール
    @{
*/

/*!
    @brief      拡張情報付きプロフィールリストを取得します。

    @param[out] outAsync    非同期コンテキスト。
    @param[out] outProfiles プロフィールリスト。
    @param[in]  uid         ユーザーアカウント。
    @param[in]  accountIds  任意のユーザーのネットワークサービスアカウント ID リスト。
    @param[in]  count       プロフィールリストの要素数。

    @return     処理結果。

    @pre
        - outAsync != nullptr
        - outProfiles != nullptr
        - 0 < count <= GetProfileCountMax
        - 指定したユーザーアカウントが Open 状態である。
        - NIFM ライブラリを初期化している。

    @details
                指定したネットワークサービスアカウント ID に対応するユーザーのプロフィール情報を取得します。@n
                指定したネットワークサービスアカウント ID に対応するユーザーが存在しない場合、該当する @ref ProfileExtra オブジェクトはクリアされます。

                拡張プロフィールには公開範囲が設定できるため、指定したユーザーアカウントと相手の関係性によって取得できる情報は変化します。

                本関数は通信処理を行うため、事前にインターネット利用要求を行い、それが受理されている必要があります。@n
                通信処理のエラーハンドリングは、 @ref AsyncContext::GetResult を参照してください。
*/
nn::Result GetProfileExtraList(AsyncContext* outAsync, ProfileExtra* outProfiles,
    const nn::account::Uid& uid, const nn::account::NetworkServiceAccountId* accountIds, int count) NN_NOEXCEPT;

/*!
    @brief      自分と指定した相手の関係を取得します。

    @param[out] outAsync        非同期コンテキスト。
    @param[out] outRelationship 自分と相手ユーザー間の関係。
    @param[in]  uid             ユーザーアカウント。
    @param[in]  accountId       相手のネットワークサービスアカウント ID。

    @return     処理結果。

    @pre
        - outAsync != nullptr
        - outRelationship != nullptr
        - 指定したユーザーアカウントが Open 状態である。
        - NIFM ライブラリを初期化している。

    @details
                自分と相手の関係を取得することで、以下のようなシチュエーションを事前に防ぐことができるようになります。

                - フレンドをブロックしてしまった。
                - ブロックした相手にフレンド申請してしまった。

                本関数は通信処理を行うため、事前にインターネット利用要求を行い、それが受理されている必要があります。@n
                通信処理のエラーハンドリングは、 @ref AsyncContext::GetResult を参照してください。
*/
nn::Result GetRelationship(AsyncContext* outAsync, Relationship* outRelationship,
    const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT;

/*!
    @}
    @name ユーザープレゼンス
    @{
*/

/*!
    @brief      ユーザープレゼンスビューを取得します。

    @param[out] outPresence ユーザープレゼンスビュー。
    @param[in]  uid         ユーザーアカウント。

    @return     処理結果。

    @pre
        - outPresence != nullptr
        - 指定したユーザーアカウントが Open 状態である。
*/
nn::Result GetUserPresenceView(UserPresenceView* outPresence,
    const nn::account::Uid& uid) NN_NOEXCEPT;

/*!
    @}
    @name ユーザー設定
    @{
*/

/*!
    @brief      本体ストレージ上のユーザー設定のキャッシュを最新の状態に更新します。

    @param[out] outAsync    非同期コンテキスト。
    @param[in]  uid         ユーザーアカウント。

    @return     処理結果。

    @pre
        - outAsync != nullptr
        - 指定したユーザーアカウントが Open 状態である。
        - NIFM ライブラリを初期化している。

    @details
                本関数は、任意のタイミングでサーバーと同期処理を行います。@n
                通常、ユーザーアカウント選択時にバックグラウンドで同期処理が行われます。

                通信処理のエラーハンドリングは、 @ref AsyncContext::GetResult を参照してください。

    @see
        - UserSetting
*/
nn::Result SyncUserSetting(AsyncContext* outAsync,
    const nn::account::Uid& uid) NN_NOEXCEPT;

/*!
    @}
    @name フレンドコード
    @{
*/

/*!
    @brief      フレンドコードから拡張情報付きユーザープロフィールを取得します。

    @param[out] outAsync    非同期コンテキスト。
    @param[out] outProfile  プロフィール。
    @param[in]  uid         ユーザーアカウント。
    @param[in]  friendCode  フレンドコード。

    @return     処理結果。

    @pre
        - outAsync != nullptr
        - outProfile != nullptr
        - 指定したユーザーアカウントが Open 状態である。
        - NIFM ライブラリを初期化している。

    @details
                ユーザーに見えるフレンドコードは以下のフォーマットです。

                - ［接頭辞］－［数字 4 桁］－［数字 4 桁］－［数字 4 桁］

                本関数に指定するフレンドコード文字列は、接頭辞を除く数字列とハイフンです。

                フレンドコードはユーザーが入力する文字列であるため、システムは厳密なフォーマットチェックは行いません。@n
                ハイフンなし数字 12 桁の指定や、数字 3 桁区切り指定は問題ありません。

                拡張プロフィールには公開範囲が設定できるため、指定したユーザーアカウントと相手の関係性によって取得できる情報は変化します。

                本関数は通信処理を行うため、事前にインターネット利用要求を行い、それが受理されている必要があります。@n
                通信処理のエラーハンドリングは、 @ref AsyncContext::GetResult を参照してください。

                非同期処理の結果が ResultInvalidFriendCodeFormat の場合、入力内容の確認を行ってください。@n
                非同期処理の結果が ResultNetworkServiceAccountNotExists の場合、入力したフレンドコードに該当するユーザーは存在しません。
*/
nn::Result GetProfileExtra(AsyncContext* outAsync, ProfileExtra* outProfile,
    const nn::account::Uid& uid, const FriendCode& friendCode) NN_NOEXCEPT;

/*!
    @}
    @name いっしょに遊んだユーザー
    @{
*/

/*!
    @brief      いっしょに遊んだユーザーを取得します。

    @param[out] outCount        取得した数。
    @param[out] outHistories    いっしょに遊んだユーザー。
    @param[in]  uid             ユーザーアカウント。
    @param[in]  offset          オフセット。
    @param[in]  count           いっしょに遊んだユーザーの要素数。

    @return     処理結果。

    @pre
        - outCount != nullptr
        - outHistories != nullptr
        - offset >= 0
        - count > 0
        - 指定したユーザーアカウントが Open 状態である。

    @details
                指定したオフセットから指定した数のいっしょに遊んだユーザー情報を取得します。

                いっしょに遊んだユーザー情報は本体ストレージ上に保存されるため、インフラ未接続時も取得できます。
*/
nn::Result GetPlayHistoryList(int* outCount, PlayHistory* outHistories,
    const nn::account::Uid& uid, int offset, int count) NN_NOEXCEPT;

/*!
    @brief      いっしょに遊んだユーザーの統計情報を取得します。

    @param[out] outStatistics   統計情報。
    @param[in]  uid             ユーザーアカウント。

    @return     処理結果。

    @pre
        - outStatistics != nullptr
        - 指定したユーザーアカウントが Open 状態である。
*/
nn::Result GetPlayHistoryStatistics(PlayHistoryStatistics* outStatistics,
    const nn::account::Uid& uid) NN_NOEXCEPT;

/*!
    @brief      いっしょに遊んだユーザー情報を削除します。

    @param[in]  uid ユーザーアカウント。

    @return     処理結果。

    @pre
        - 指定したユーザーアカウントが Open 状態である。

    @details
                いっしょに遊んだユーザー情報は、本体ストレージ上にマスターデータを持ちます。@n
                ニンテンドーアカウントの紐付け解除時にキャッシュが削除されますが、この時点でいっしょに遊んだユーザー情報は削除されません。
*/
nn::Result DeletePlayHistory(const nn::account::Uid& uid) NN_NOEXCEPT;

/*!
    @}
    @name オーバーレイ通知
    @{
*/

/*!
    @brief      リスト情報のサマリーを表示するオーバーレイ通知の発行を要求します。

    @return     処理結果。

    @details
                フレンドシステムは、ユーザーアカウントの Open 状態への変化を検知してリスト情報のサマリーをオーバーレイ通知します。@n
                本関数は、任意のタイミングでオーバーレイ通知の発行を要求することができます。

                本関数は、以下のタイミングで利用されることを想定しています。

                - ユーザーアカウントを Open 状態にしたまま本体を長期間スリープし、そこから復帰したタイミング。@n
                  この場合、ユーザーアカウントはすでに Open 状態であるため、フレンドシステムは自動でオーバーレイ通知しません。
*/
nn::Result RequestListSummaryOverlayNotification() NN_NOEXCEPT;

/*!
    @}
    @name システム状態管理
    @{
*/

/*!
    @brief      システムの自律動作を停止します。

    @param[out] outSuspension   自律動作停止の期間管理オブジェクト。

    @return     処理結果。

    @pre
        - outSuspension != nullptr

    @details
                自律動作を停止することにより、バックグラウンドでサーバーと通信することを抑制できます。@n
                これにより、サーバーと通信して得られたデータのキャッシュ書き込みも抑制されます。

                ただし、以下の場合、通信やファイル書き込みが行われる可能性があります。

                - アプリケーションが非同期 API を呼び出す
                - ユーザーアカウントの Open/Close をトリガーにしたプレイ履歴の更新

                停止した自律動作を再開するには、 DaemonSuspension オブジェクトを破棄してください。
*/
nn::Result SuspendDaemon(DaemonSuspension* outSuspension) NN_NOEXCEPT;

/*!
    @}
    @name アカウント管理
    @{
*/

/*!
    @brief      ネットワークサービスアカウントに関するキャッシュを削除します。

    @param[in]  uid ユーザーアカウント。

    @return     処理結果。

    @pre
        - 指定したユーザーアカウントが Open 状態である。

    @details
                フレンドライブラリ管理の本体ストレージ上に保存されたネットワークサービスアカウントに関するキャッシュデータを削除します。@n
                本関数は、ニンテンドーアカウントの紐付け解除時に呼び出してください。

                いっしょに遊んだユーザーはユーザーアカウントが持つデータなので、本関数で削除されることはありません。
*/
nn::Result DeleteNetworkServiceAccountCache(const nn::account::Uid& uid) NN_NOEXCEPT;

/*!
    @}
    @name 経緯情報の拡張情報
    @{
*/

/*!
    @brief      外部アプリケーションカタログを取得します。

    @param[out] outAsync    非同期コンテキスト。
    @param[out] outCatalog  外部アプリケーションカタログ。
    @param[in]  catalogId   外部アプリケーションカタログ ID。
    @param[in]  language    言語コード。

    @return     処理結果。

    @pre
        - outAsync != nullptr
        - outCatalog != nullptr
        - NIFM ライブラリを初期化している。

    @details
                外部アプリケーションカタログ ID は、外部アプリケーション上でいっしょに遊んだことのあるのユーザーがフレンド候補に出現した際、取得できます。

                本関数は通信処理を行うため、事前にインターネット利用要求を行い、それが受理されている必要があります。@n
                通信処理のエラーハンドリングは、 @ref AsyncContext::GetResult を参照してください。
*/
nn::Result GetExternalApplicationCatalog(AsyncContext* outAsync, ExternalApplicationCatalog* outCatalog,
    const ExternalApplicationCatalogId& catalogId, const nn::settings::LanguageCode language) NN_NOEXCEPT;

/*!
    @brief      Mii 画像 URL を取得します。

    @param[out] outUrl              Mii 画像 URL。
    @param[in]  miiImageUrlParam    Mii 画像 URL パラメータ。

    @return     処理結果。

    @pre
        - outUrl != nullptr

    @details
                @ref GetNintendoNetworkIdInfo で取得した Mii 画像 URL パラメータを指定することで、 Mii 画像 URL を取得できます。
*/
nn::Result GetMiiImageUrl(Url* outUrl, const MiiImageUrlParam& miiImageUrlParam) NN_NOEXCEPT;

/*!
    @}
*/

}}
