﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/friends/detail/service/json/friends_Json.h>
#include <nn/friends/detail/service/json/friends_JsonPath.h>
#include <nn/friends/detail/friends_Log.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace json {

/*!
    @brief      JSON イベントハンドラです。

    @details
                各 JSON データのイベントハンドルに特化した処理を記述します。@n
                JSON パーサーの利用者は、本クラスの派生を定義し、処理を記述してください。
*/
class JsonEventHandler
{
public:
    /*!
        @brief      コンストラクタです。
    */
    JsonEventHandler() NN_NOEXCEPT {}

    /*!
        @brief      デストラクタです。
    */
    virtual ~JsonEventHandler() NN_NOEXCEPT {}

    /*!
        @brief      オブジェクトの開始イベントです。

        @param[in]  jsonPath    JSON パス。

        @return     処理を継続するかどうか。
    */
    virtual bool OnStartObject(const JsonPath& jsonPath) NN_NOEXCEPT
    {
        NN_UNUSED(jsonPath);

        return true;
    }

    /*!
        @brief      オブジェクトの終了イベントです。

        @param[in]  jsonPath    JSON パス。
        @param[in]  numObjects  オブジェクト数。

        @return     処理を継続するかどうか。
    */
    virtual bool OnEndObject(const JsonPath& jsonPath, size_t numObjects) NN_NOEXCEPT
    {
        NN_UNUSED(jsonPath);
        NN_UNUSED(numObjects);

        return true;
    }

    /*!
        @brief      配列の開始イベントです。

        @param[in]  jsonPath    JSON パス。

        @return     処理を継続するかどうか。
    */
    virtual bool OnStartArray(const JsonPath& jsonPath) NN_NOEXCEPT
    {
        NN_UNUSED(jsonPath);

        return true;
    }

    /*!
        @brief      配列の終了イベントです。

        @param[in]  jsonPath    JSON パス。
        @param[in]  numElements 要素数。

        @return     処理を継続するかどうか。
    */
    virtual bool OnEndArray(const JsonPath& jsonPath, size_t numElements) NN_NOEXCEPT
    {
        NN_UNUSED(jsonPath);
        NN_UNUSED(numElements);

        return true;
    }

    /*!
        @brief      キーの出現イベントです。

        @param[in]  jsonPath    JSON パス。

        @return     処理を継続するかどうか。
    */
    virtual bool OnKey(const JsonPath& jsonPath, const char* key) NN_NOEXCEPT
    {
        NN_UNUSED(jsonPath);
        NN_UNUSED(key);

        return true;
    }

    /*!
        @brief      値（null）の出現イベントです。

        @param[in]  jsonPath    JSON パス。

        @return     処理を継続するかどうか。
    */
    virtual bool OnNull(const JsonPath& jsonPath, std::nullptr_t) NN_NOEXCEPT
    {
        NN_UNUSED(jsonPath);

        return true;
    }

    /*!
        @brief      値（bool）の出現イベントです。

        @param[in]  jsonPath    JSON パス。
        @param[in]  value       値。

        @return     処理を継続するかどうか。
    */
    virtual bool OnBoolean(const JsonPath& jsonPath, bool value) NN_NOEXCEPT
    {
        NN_UNUSED(jsonPath);
        NN_UNUSED(value);

        return true;
    }

    /*!
        @brief      値（integer）の出現イベントです。

        @param[in]  jsonPath    JSON パス。
        @param[in]  value       値。

        @return     処理を継続するかどうか。
    */
    virtual bool OnInteger(const JsonPath& jsonPath, const AnyInteger& value) NN_NOEXCEPT
    {
        NN_UNUSED(jsonPath);
        NN_UNUSED(value);

        return true;
    }

    /*!
        @brief      値（double）の出現イベントです。

        @param[in]  jsonPath    JSON パス。
        @param[in]  value       値。

        @return     処理を継続するかどうか。
    */
    virtual bool OnDouble(const JsonPath& jsonPath, double value) NN_NOEXCEPT
    {
        NN_UNUSED(jsonPath);
        NN_UNUSED(value);

        return true;
    }

    /*!
        @brief      値（string）の出現イベントです。

        @param[in]  jsonPath        JSON パス。
        @param[in]  value           値。
        @param[in]  length          文字列長。
        @param[in]  isOverflowed    オーバーフローフラグ。

        @return     処理を継続するかどうか。
    */
    virtual bool OnString(const JsonPath& jsonPath, const char* value, size_t length, bool isOverflowed) NN_NOEXCEPT
    {
        NN_UNUSED(jsonPath);
        NN_UNUSED(value);
        NN_UNUSED(length);
        NN_UNUSED(isOverflowed);

        return true;
    }

    /*!
        @brief      処理結果を取得します。

        @return     処理結果。
    */
    virtual nn::Result GetResult() const NN_NOEXCEPT
    {
        NN_RESULT_SUCCESS;
    }
};

/*!
    @brief      ログ出力するイベントハンドラです。
*/
class JsonPrintEventHandler : public JsonEventHandler
{
public:
    virtual bool OnStartArray(const JsonPath& jsonPath) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(jsonPath);
        NN_DETAIL_FRIENDS_INFO("[friends] %s:\n    [\n", jsonPath.ToString());

        return true;
    }

    virtual bool OnEndArray(const JsonPath& jsonPath, size_t numElements) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(jsonPath);
        NN_UNUSED(numElements);
        NN_DETAIL_FRIENDS_INFO("[friends] %s:\n    ] num = %zu\n", jsonPath.ToString(), numElements);

        return true;
    }

    virtual bool OnNull(const JsonPath& jsonPath, std::nullptr_t) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(jsonPath);
        NN_DETAIL_FRIENDS_INFO("[friends] %s:\n    value = null\n", jsonPath.ToString());

        return true;
    }

    virtual bool OnBoolean(const JsonPath& jsonPath, bool value) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(jsonPath);
        NN_UNUSED(value);
        NN_DETAIL_FRIENDS_INFO("[friends] %s:\n    value = %s\n", jsonPath.ToString(), value ? "true" : "false");

        return true;
    }

    virtual bool OnInteger(const JsonPath& jsonPath, const AnyInteger& value) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(jsonPath);
        NN_UNUSED(value);

        if (value.isSigned)
        {
            NN_DETAIL_FRIENDS_INFO("[friends] %s:\n    value = %lld\n", jsonPath.ToString(), value.s);
        }
        else
        {
            NN_DETAIL_FRIENDS_INFO("[friends] %s:\n    value = %llu\n", jsonPath.ToString(), value.u);
        }

        return true;
    }

    virtual bool OnDouble(const JsonPath& jsonPath, double value) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(jsonPath);
        NN_UNUSED(value);
        NN_DETAIL_FRIENDS_INFO("[friends] %s:\n    value = %f\n", jsonPath.ToString(), value);

        return true;
    }

    virtual bool OnString(const JsonPath& jsonPath, const char* value, size_t length, bool isOverflowed) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(jsonPath);
        NN_UNUSED(value);
        NN_UNUSED(length);
        NN_UNUSED(isOverflowed);
        NN_DETAIL_FRIENDS_INFO("[friends] %s:\n    value = \"%s\"\n", jsonPath.ToString(), value);

        return true;
    }
};

}}}}}
