﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/friends/detail/service/friends_Common.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

/*!
    @brief      ユーザー設定の管理モジュールです。
*/
class UserSettingManager
{
private:
    NN_DISALLOW_COPY(UserSettingManager);
    NN_DISALLOW_MOVE(UserSettingManager);

public:
    /*!
        @brief      レコードです。
    */
    struct Record
    {
        // 4
        int32_t presencePermission;
        // 4
        int32_t playLogPermission;
        // 1
        bool friendRequestReception;
        // 7
        char reserved1[7];
        // 32
        FriendCode friendCode;
        // 8
        nn::time::PosixTime friendCodeNextIssuableTime;
        // 1024
        ProfileExtraImpl profileExtra;
        // 968
        char reserved2[968];
    };

    NN_STATIC_ASSERT(sizeof (Record) == 2048);

private:
    /*!
        @brief      コンストラクタです。
    */
    UserSettingManager() NN_NOEXCEPT;

public:
    /*!
        @brief      インスタンスを取得します。

        @return     インスタンス。
    */
    static UserSettingManager& GetInstance() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(UserSettingManager, s_Instance);
        return s_Instance;
    }

public:
    /*!
        @brief      プレイログを取得します。

        @param[out] outCount    取得した数。
        @param[out] outPlayLogs プレイログ。
        @param[in]  uid         ユーザーアカウント。
        @param[in]  count       プレイログの要素数。

        @return     処理結果。

        @pre
            - outCount != nullptr
            - outPlayLogs != nullptr
            - count > 0
    */
    nn::Result GetPlayLogList(int* outCount, PlayLog* outPlayLogs,
        const nn::account::Uid& uid, int count) NN_NOEXCEPT;

    /*!
        @brief      プレイログの公開範囲を取得します。

        @param[out] outPermission   プレイログの公開範囲。
        @param[in]  uid             ユーザーアカウント。

        @return     処理結果。

        @pre
            - outPermission != nullptr
    */
    nn::Result GetPlayLogPermission(ExtraPermission* outPermission, const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      フレンドコードを保持しているかどうかを確認します。

        @param[out] out フレンドコードを保持しているかどうか。
        @param[in]  uid ユーザーアカウント。

        @return     処理結果。

        @pre
            - out != nullptr
    */
    nn::Result HasFriendCode(bool* out, const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      ユーザー設定を読み込みます。

        @param[out] outSetting  ユーザー設定。
        @param[in]  uid         ユーザーアカウント。

        @return     処理結果。

        @pre
            - outSetting != nullptr
    */
    nn::Result Read(UserSettingImpl* outSetting, const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      ユーザー設定を書き込みます。

        @param[in]  uid         ユーザーアカウント。
        @param[in]  resource    ユーザーリソース。

        @return     処理結果。
    */
    nn::Result Write(const nn::account::Uid& uid, const UserResource& resource) NN_NOEXCEPT;

    /*!
        @brief      メモリキャッシュを無効化します。
    */
    void Invalidate() NN_NOEXCEPT;

private:
    //
    nn::os::Mutex m_Mutex;
    //
    nn::account::Uid m_CurrentUid;
    //
    Record m_Record;

private:
    //
    nn::Result Load(const nn::account::Uid& uid) NN_NOEXCEPT;
    nn::Result LoadImpl(const nn::account::Uid& uid) NN_NOEXCEPT;
    //
    void NormalizePlayLogPermission() NN_NOEXCEPT;
};

}}}}}
