﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/friends/detail/service/friends_Common.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

/*!
    @brief      プロフィール画像の管理モジュールです。
*/
class ImageManager
{
private:
    NN_DISALLOW_COPY(ImageManager);
    NN_DISALLOW_MOVE(ImageManager);

public:
    /*!
        @brief      レコードの最大数です。

        @details
                    全ユーザーの全フレンドのレコードが載るサイズです。
    */
    static const int RecordCountMax = nn::account::UserCountMax * FriendCountMax;

    /*!
        @brief      URL のハッシュ値です。
    */
    struct UrlHash
    {
        Bit8 value[7];
    };

    /*!
        @brief      レコードです。
    */
    struct Record
    {
        // 8
        nn::account::NetworkServiceAccountId accountId;
        // 1
        bool isDeleted;
        // 7
        UrlHash hash;
    };

    NN_STATIC_ASSERT(sizeof (Record) == 16);

private:
    /*!
        @brief      コンストラクタです。
    */
    ImageManager() NN_NOEXCEPT;

public:
    /*!
        @brief      インスタンスを取得します。

        @return     インスタンス。
    */
    static ImageManager& GetInstance() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(ImageManager, s_Instance);
        return s_Instance;
    }

public:
    /*!
        @brief      プロフィール画像を保存します。

        @param[in]  accountId   ネットワークサービスアカウント ID。
        @param[in]  url         URL。
        @param[in]  image       プロフィール画像。
        @param[in]  imageSize   プロフィール画像のサイズ。

        @return     処理結果。

        @pre
            - image != nullptr
            - imageSize > 0

        @details
                    ストレージに空きがない場合、古いプロフィール画像を削除します。
    */
    nn::Result SaveImage(nn::account::NetworkServiceAccountId accountId, const Url& url,
        const void* image, size_t imageSize) NN_NOEXCEPT;

    /*!
        @brief      プロフィール画像を読み込みます。

        @param[out] outSize     読み込んだサイズ。
        @param[in]  accountId   ネットワークサービスアカウント ID。
        @param[in]  buffer      バッファ。
        @param[in]  size        バッファサイズ。

        @return     処理結果。

        @pre
            - outSize != nullptr
            - buffer != nullptr
            - size > 0
    */
    nn::Result LoadImage(size_t* outSize, nn::account::NetworkServiceAccountId accountId,
        void* buffer, size_t size) NN_NOEXCEPT;

    /*!
        @brief      指定したネットワークサービスアカウントのプロフィール画像を追加、または、更新可能かどうかを確認します。

        @param[in]  accountId   ネットワークサービスアカウント ID。
        @param[in]  url         URL。

        @return     追加、または、更新可能かどうか。
    */
    bool CanAddOrUpdate(nn::account::NetworkServiceAccountId accountId, const Url& url) NN_NOEXCEPT;

    /*!
        @brief      指定したネットワークサービスアカウントリストのレコードリスト内順序を先頭に移動します。

        @param[in]  accountIds  ネットワークサービスアカウント ID リスト。
        @param[in]  count       ネットワークサービスアカウント ID リストの要素数。

        @pre
            - accountIds != nullptr
            - count > 0

        @details
                    順序を上にあげることで、ストレージ容量不足時に削除される可能性が低くなります。@n
                    本関数はフレンドリストの取得等、クライアントがフレンド情報にアクセスする際に呼び出してください。

                    本関数は、高速化のためセーブデータへの書き込みは行いません。
    */
    void MoveToTopRecords(const nn::account::NetworkServiceAccountId* accountIds, int count) NN_NOEXCEPT;

    /*!
        @brief      指定したフレンドリストのレコードリスト内順序を先頭に移動します。

        @param[in]  friends フレンドリスト。
        @param[in]  count   フレンドリストの要素数。

        @pre
            - friends != nullptr
            - count > 0

        @details
                    順序を上にあげることで、ストレージ容量不足時に削除される可能性が低くなります。@n
                    本関数はフレンドリストの取得等、クライアントがフレンド情報にアクセスする際に呼び出してください。

                    本関数は、高速化のためセーブデータへの書き込みは行いません。
    */
    void MoveToTopRecords(const FriendImpl* friends, int count) NN_NOEXCEPT;

private:
    //
    nn::os::Mutex m_Mutex;
    //
    Record m_Records[RecordCountMax];
    int m_Count;
    //
    bool m_IsLoaded;

private:
    //
    void CreateRecord(Record* outRecord, nn::account::NetworkServiceAccountId accountId, const Url& url) NN_NOEXCEPT;
    void InsertRecord(const Record& record) NN_NOEXCEPT;
    void MoveToTopRecord(nn::account::NetworkServiceAccountId accountId, bool allowDeleted) NN_NOEXCEPT;
    void RemoveRecord(nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT;
    void RemoveOldRecord() NN_NOEXCEPT;
    //
    nn::Result LoadRecordWithLockStorage() NN_NOEXCEPT;
    nn::Result LoadRecord() NN_NOEXCEPT;
    nn::Result LoadRecordImpl() NN_NOEXCEPT;
    nn::Result SaveRecord() NN_NOEXCEPT;
    //
    int SearchRecord(nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT;
    //
    void MakeImagePath(char* buffer, size_t size, const Record& record) NN_NOEXCEPT;
    //
    nn::Result EnsureCapacity(int64_t requiredSize) NN_NOEXCEPT;
    //
    nn::Result DeleteImage(nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT;
    nn::Result DeleteOldImage() NN_NOEXCEPT;
};

}}}}}
