﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/friends/detail/service/friends_Common.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

/*!
    @brief      フレンド申請件数の管理モジュールです。
*/
class FriendRequestCountManager
{
private:
    NN_DISALLOW_COPY(FriendRequestCountManager);
    NN_DISALLOW_MOVE(FriendRequestCountManager);

private:
    /*!
        @brief      コンストラクタです。
    */
    FriendRequestCountManager() NN_NOEXCEPT;

public:
    /*!
        @brief      インスタンスを取得します。

        @return     インスタンス。
    */
    static FriendRequestCountManager& GetInstance() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(FriendRequestCountManager, s_Instance);
        return s_Instance;
    }

public:
    /*!
        @brief      受信したフレンド申請件数を取得します。

        @param[out] outUnreadCount  未読件数。
        @param[out] outReadCount    既読件数。
        @param[in]  uid             ユーザーアカウント。

        @return     処理結果。

        @pre
            - outUnreadCount != nullptr
            - outReadCount != nullptr
    */
    nn::Result GetCount(int* outUnreadCount, int* outReadCount,
        const nn::account::Uid& uid) const NN_NOEXCEPT;

    /*!
        @brief      受信したフレンド申請件数を設定します。

        @param[in]  uid         ユーザーアカウント。
        @param[in]  unreadCount 未読件数。
        @param[in]  readCount   既読件数。
    */
    nn::Result SetCount(const nn::account::Uid& uid, int unreadCount, int readCount) NN_NOEXCEPT;

    /*!
        @brief      ユーザーアカウントの追加イベントを通知します。

        @param[in]  uid ユーザーアカウント。
    */
    void NotifyUserAdded(const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      ユーザーアカウントの削除イベントを通知します。

        @param[in]  uid ユーザーアカウント。
    */
    void NotifyUserRemoved(const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      ネットワークサービスアカウントの有効性の変化イベントを通知します。

        @param[in]  uid         ユーザーアカウント。
        @param[in]  isAvailable 有効かどうか。
    */
    void NotifyNetworkServiceAccountAvailabilityChanged(const nn::account::Uid& uid, bool isAvailable) NN_NOEXCEPT;

private:
    //
    struct Record
    {
        nn::account::Uid uid;
        int unreadCount;
        int readCount;
    };

private:
    //
    mutable nn::os::Mutex m_Mutex;
    //
    Record m_Records[nn::account::UserCountMax];

private:
    //
    int FindUser(const nn::account::Uid& uid) const NN_NOEXCEPT;
};

}}}}}
