﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/friends/detail/service/friends_Common.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

/*!
    @brief      フレンドリストの管理モジュールです。
*/
class FriendListManager
{
private:
    NN_DISALLOW_COPY(FriendListManager);
    NN_DISALLOW_MOVE(FriendListManager);

public:
    /*!
        @brief      レコードです。
    */
    struct Record
    {
        // 8
        nn::account::NetworkServiceAccountId accountId;
        // 40
        nn::account::Nickname nickname;
        char nicknamePadding[7];
        // 160
        Url profileImageUrl;
        // 800
        PlayLog playLog[PlayLogCountMax];
        // 168
        PlayRecord lastPlayRecord;
        // 4
        int32_t requestType;
        // 1
        bool isFavorite;
        // 1
        bool isNewly;
        // 1
        bool isOnlineNotification;
        // 1
        Bit8 reseved1[1];
        // 96
        RouteInfo requestRouteInfo;
        // 1
        bool isNewlyConfirmed;
        // 1
        bool isLastPlayRecordUpdated;
        // 6
        char reserved2[6];
        // 48
        RouteInfoExtra requestRouteInfoExtra;
        // 200
        char reserved3[200];
    };

    NN_STATIC_ASSERT(sizeof (Record) == 1536);

    /*!
        @brief      フレンド人数のキャッシュです。
    */
    struct FriendCountCache
    {
        nn::account::Uid uid;
        int32_t total;
        int32_t newlyArrived;
        bool isSet;

    public:
        void Initialize(const nn::account::Uid& uid_) NN_NOEXCEPT
        {
            uid = uid_;
            total = 0;
            newlyArrived = 0;
            isSet = false;
        }

        void Set(int32_t total_, int32_t newlyArrived_) NN_NOEXCEPT
        {
            total = total_;
            newlyArrived = newlyArrived_;
            isSet = true;
        }

        void ResetCount() NN_NOEXCEPT
        {
            total = 0;
            newlyArrived = 0;
            isSet = false;
        }
    };

private:
    /*!
        @brief      コンストラクタです。
    */
    FriendListManager() NN_NOEXCEPT;

public:
    /*!
        @brief      インスタンスを取得します。

        @return     インスタンス。
    */
    static FriendListManager& GetInstance() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(FriendListManager, s_Instance);
        return s_Instance;
    }

public:
    /*!
        @brief      フレンドリストを取得します。

        @param[out] outCount        取得した数。
        @param[out] outAccountIds   フレンドリスト。
        @param[in]  uid             ユーザーアカウント。
        @param[in]  offset          オフセット。
        @param[in]  count           フレンドリストの要素数。
        @param[in]  filter          フレンドリスト取得のフィルター。
        @param[in]  appInfo         アプリケーション情報。

        @return     処理結果。

        @pre
            - outCount != nullptr
            - outAccountIds != nullptr
    */
    nn::Result GetFriendList(int* outCount, nn::account::NetworkServiceAccountId* outAccountIds,
        const nn::account::Uid& uid, int offset, int count, const detail::ipc::SizedFriendFilter& filter,
        const ApplicationInfo& appInfo) NN_NOEXCEPT;

    /*!
        @brief      フレンドリストを取得します。

        @param[out] outCount                取得した数。
        @param[out] outFriends              フレンドリスト。
        @param[in]  uid                     ユーザーアカウント。
        @param[in]  offset                  オフセット。
        @param[in]  count                   フレンドリストの要素数。
        @param[in]  filter                  フレンドリスト取得のフィルター。
        @param[in]  appInfo                 アプリケーション情報。
        @param[in]  checkAppFieldPermission アプリケーション固有情報の取得権限をチェックするかどうか。

        @return     処理結果。

        @pre
            - outCount != nullptr
            - outFriends != nullptr
    */
    nn::Result GetFriendList(int* outCount, FriendImpl* outFriends,
        const nn::account::Uid& uid, int offset, int count, const detail::ipc::SizedFriendFilter& filter,
        const ApplicationInfo& appInfo, bool checkAppFieldPermission) NN_NOEXCEPT;

    /*!
        @brief      フレンドの人数を取得します。

        @param[out] outCount    条件に合致したフレンドの人数。
        @param[in]  uid         ユーザーアカウント。
        @param[in]  filter      フレンドリスト取得のフィルター。
        @param[in]  appInfo     アプリケーション情報。

        @return     処理結果。

        @pre
            - outCount != nullptr
    */
    nn::Result GetFriendCount(int* outCount,
        const nn::account::Uid& uid, const detail::ipc::SizedFriendFilter& filter, const ApplicationInfo& appInfo) NN_NOEXCEPT;

    /*!
        @brief      新着フレンドの人数を取得します。

        @param[out] outCount    新着フレンドの人数。
        @param[in]  uid         ユーザーアカウント。

        @return     処理結果。

        @pre
            - outCount != nullptr
    */
    nn::Result GetNewlyFriendCount(int* outCount,
        const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      指定したフレンドのフレンド情報を取得します。

        @param[out] outFriends              フレンドリスト。
        @param[in]  uid                     ユーザーアカウント。
        @param[in]  accountIds              フレンドのネットワークサービスアカウント ID リスト。
        @param[in]  count                   フレンドのネットワークサービスアカウント ID リストの要素数。
        @param[in]  appInfo                 アプリケーション情報。
        @param[in]  checkAppFieldPermission アプリケーション固有情報の取得権限をチェックするかどうか。

        @return     処理結果。

        @pre
            - outFriends != nullptr
            - accountIds != nullptr
    */
    nn::Result GetFriendInfo(FriendImpl* outFriends,
        const nn::account::Uid& uid, const nn::account::NetworkServiceAccountId* accountIds, int count,
        const ApplicationInfo& appInfo, bool checkAppFieldPermission) NN_NOEXCEPT;

    /*!
        @brief      フレンドの詳細情報を取得します。

        @param[out] outInfo     フレンドの詳細情報。
        @param[in]  uid         ユーザーアカウント。
        @param[in]  accountId   フレンドのネットワークサービスアカウント ID。

        @return     処理結果。

        @pre
            - outInfo != nullptr
    */
    nn::Result GetFriendDetailedInfo(FriendDetailedInfoImpl* outInfo,
        const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT;

    /*!
        @brief      ローカルで更新されたレコードを取得します。

        @param[out] outCount    取得した数。
        @param[out] outRecords  レコードリスト。
        @param[in]  uid         ユーザーアカウント。
        @param[in]  count       レコードリストの要素数。

        @return     処理結果。

        @pre
            - outCount != nullptr
            - outRecords != nullptr
    */
    nn::Result GetLocalUpdatedRecord(int* outCount, Record* outRecords,
        const nn::account::Uid& uid, int count) NN_NOEXCEPT;

    /*!
        @brief      フレンド設定を取得します。

        @param[out] outSetting  フレンド設定。
        @param[in]  uid         ユーザーアカウント。
        @param[in]  accountId   フレンドのネットワークサービスアカウント ID。

        @return     処理結果。

        @pre
            - outSetting != nullptr
    */
    nn::Result GetFriendSetting(FriendSettingImpl* outSetting,
        const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT;

    /*!
        @brief      新着マークを落とします。

        @param[in]  uid         ユーザーアカウント。
        @param[in]  accountId   フレンドのネットワークサービスアカウント ID。

        @return     処理結果。
    */
    nn::Result DropNewly(const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT;

    /*!
        @brief      新着フレンド全員の新着マークを落とします。

        @param[in]  uid ユーザーアカウント。

        @return     処理結果。
    */
    nn::Result DropNewlyAll(const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      最後にいっしょに遊んだ記録を更新します。

        @param[in]  uid             ユーザーアカウント。
        @param[in]  accountId       フレンドのネットワークサービスアカウント ID。
        @param[in]  lastPlayRecord  最後にいっしょに遊んだ記録。

        @return     処理結果。
    */
    nn::Result UpdateLastPlayRecord(const nn::account::Uid& uid,
        nn::account::NetworkServiceAccountId accountId, const PlayRecord& lastPlayRecord) NN_NOEXCEPT;

    /*!
        @brief      フレンドを更新します。

        @param[in]  uid                     ユーザーアカウント。
        @param[in]  resources               フレンドリソース。
        @param[in]  isLocalChangeApplied    ローカルの変更をサーバーに適用したかどうか。

        @return     処理結果。
    */
    nn::Result UpdateFriend(const nn::account::Uid& uid, const FriendResource& resource, bool isLocalChangeApplied) NN_NOEXCEPT;

    /*!
        @brief      フレンドリストを更新します。

        @param[in]  uid         ユーザーアカウント。
        @param[in]  resources   フレンドリソースリスト。
        @param[in]  count       フレンドリソースリストの要素数。

        @return     処理結果。

        @pre
            - resources != nullptr

        @details
                    本関数を呼び出す前に、ローカルの変更をサーバーに適用しておく必要があります。
    */
    nn::Result UpdateFriendList(const nn::account::Uid& uid, const FriendResource* resources, int count) NN_NOEXCEPT;

    /*!
        @brief      フレンドを削除します。

        @param[in]  uid         ユーザーアカウント。
        @param[in]  accountId   フレンドのネットワークサービスアカウント ID。

        @return     処理結果。
    */
    nn::Result DeleteFriend(const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT;

    /*!
        @brief      メモリキャッシュを無効化します。
    */
    void Invalidate() NN_NOEXCEPT;

    /*!
        @brief      ユーザーアカウントの追加イベントを通知します。

        @param[in]  uid ユーザーアカウント。
    */
    void NotifyUserAdded(const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      ユーザーアカウントの削除イベントを通知します。

        @param[in]  uid ユーザーアカウント。
    */
    void NotifyUserRemoved(const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      ネットワークサービスアカウントの有効性の変化イベントを通知します。

        @param[in]  uid         ユーザーアカウント。
        @param[in]  isAvailable 有効かどうか。
    */
    void NotifyNetworkServiceAccountAvailabilityChanged(const nn::account::Uid& uid, bool isAvailable) NN_NOEXCEPT;

private:
    //
    nn::os::Mutex m_Mutex;
    //
    nn::account::Uid m_CurrentUid;
    //
    Record m_Records[FriendCountMax];
    int m_Count;
    //
    FriendCountCache m_FriendCountCache[nn::account::UserCountMax];
    //
    bool m_IsDirty;

private:
    //
    nn::Result Load(const nn::account::Uid& uid) NN_NOEXCEPT;
    nn::Result LoadImpl(const nn::account::Uid& uid) NN_NOEXCEPT;
    //
    nn::Result Save(int index) NN_NOEXCEPT;
    //
    nn::Result SaveAll() NN_NOEXCEPT;
    //
    nn::Result UpdateFriendCountCache(const nn::account::Uid& uid) NN_NOEXCEPT;
    //
    bool Filter(int index, const detail::ipc::SizedFriendFilter& filter, const ApplicationInfo& appInfo) NN_NOEXCEPT;
    //
    int SearchRecord(nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT;
    int SearchFriendCountCache(const nn::account::Uid& uid) NN_NOEXCEPT;
    //
    void GetSortIndexes(int16_t* outIndexes) NN_NOEXCEPT;
};

}}}}}
