﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/fs.h>
#include <nn/fs/fs_SystemSaveData.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

/*!
    @brief      ファイルシステムライブラリのラッパーモジュールです。
*/
class FileSystem
{
public:
    /*!
        @brief      ストレージをマウントします。

        @param[in]  mountName   マウント名。

        @return     処理結果。
    */
    static nn::Result Mount(const char* mountName) NN_NOEXCEPT;

    /*!
        @brief      ストレージをアンマウントします。

        @param[in]  mountName   マウント名。
    */
    static void Unmount(const char* mountName) NN_NOEXCEPT;

    /*!
        @brief      ストレージの空き容量を取得します。

        @param[out] outSize     空き容量。
        @param[in]  mountName   マウント名。

        @return     処理結果。

        @pre
            - outSize != nullptr
    */
    static nn::Result GetFreeSpaceSize(int64_t* outSize, const char* mountName) NN_NOEXCEPT;

    /*!
        @brief      ストレージをコミットします。

        @param[in]  mountName   マウント名。

        @return     処理結果。

        @details
                    ストレージに書き込んだファイルを確定するためには、コミットが必要です。
    */
    static nn::Result Commit(const char* mountName) NN_NOEXCEPT;

    /*!
        @brief      ディレクトリを再帰的に作成します。

        @param[in]  path        パス。
        @param[in]  isFilePath  ファイルパスを指定したかどうか。

        @return     処理結果。

        @pre
            - path != nullptr

        @details
                    ファイルパスを指定した場合、親ディレクトリの生成を行います。
    */
    static nn::Result CreateDirectoryRecursively(const char* path, bool isFilePath = false) NN_NOEXCEPT;

    /*!
        @brief      ディレクトリを再帰的に削除します。

        @param[in]  path    パス。

        @return     処理結果。

        @pre
            - path != nullptr
    */
    static nn::Result DeleteDirectoryRecursively(const char* path) NN_NOEXCEPT;

    /*!
        @brief      ファイルを新規作成します。

        @param[in]  path            パス。
        @param[in]  size            ファイルサイズ。
        @param[in]  ignoreIfExists  ファイルがすでに存在する時にエラーを返すかどうか。

        @return     処理結果。

        @pre
            - path != nullptr

        @details
                    親ディレクトリが存在しない場合、親ディレクトリも作成します。
    */
    static nn::Result CreateFile(const char* path, int64_t size, bool ignoreIfExists = true) NN_NOEXCEPT;
};

}}}}}

/*!
    @brief      ストレージを局所的にロックします。
*/
#define NN_DETAIL_FRIENDS_STORAGE_SCOPED_LOCK(mountName)        \
    detail::service::core::FileSystem::Mount(mountName);        \
    NN_UTIL_SCOPE_EXIT                                          \
    {                                                           \
        detail::service::core::FileSystem::Unmount(mountName);  \
    }

/*!
    @brief      システム情報ストレージのマウント名です。
*/
#define NN_DETAIL_FRIENDS_SYSTEM_MOUNT_NAME "friends-sys"

/*!
    @brief      プロフィール画像キャッシュストレージのマウント名です。
*/
#define NN_DETAIL_FRIENDS_IMAGE_MOUNT_NAME "friends-image"

/*!
    @brief      システム情報ストレージを局所的にロックします。
*/
#define NN_DETAIL_FRIENDS_SYSTEM_STORAGE_SCOPED_LOCK() \
    NN_DETAIL_FRIENDS_STORAGE_SCOPED_LOCK(NN_DETAIL_FRIENDS_SYSTEM_MOUNT_NAME)

/*!
    @brief      プロフィール画像キャッシュストレージを局所的にロックします。
*/
#define NN_DETAIL_FRIENDS_IMAGE_STORAGE_SCOPED_LOCK() \
    NN_DETAIL_FRIENDS_STORAGE_SCOPED_LOCK(NN_DETAIL_FRIENDS_IMAGE_MOUNT_NAME)
