﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/friends/detail/service/friends_Common.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

/*!
    @brief      ブロックリストの管理モジュールです。
*/
class BlockedUserListManager
{
private:
    NN_DISALLOW_COPY(BlockedUserListManager);
    NN_DISALLOW_MOVE(BlockedUserListManager);

public:
    /*!
        @brief      レコードです。
    */
    struct Record
    {
        BlockedUserResource resource;
    };

    NN_STATIC_ASSERT(sizeof (Record) == 312);

private:
    /*!
        @brief      コンストラクタです。
    */
    BlockedUserListManager() NN_NOEXCEPT;

public:
    /*!
        @brief      インスタンスを取得します。

        @return     インスタンス。
    */
    static BlockedUserListManager& GetInstance() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(BlockedUserListManager, s_Instance);
        return s_Instance;
    }

public:
    /*!
        @brief      ブロックリストを取得します。

        @param[out] outCount        取得した数。
        @param[out] outAccountIds   ブロックリスト。
        @param[in]  uid             ユーザーアカウント。
        @param[in]  offset          オフセット。
        @param[in]  count           ブロックリストの要素数。

        @return     処理結果。

        @pre
            - outCount != nullptr
            - outAccountIds != nullptr
    */
    nn::Result GetBlockedUserList(int* outCount, nn::account::NetworkServiceAccountId* outAccountIds,
        const nn::account::Uid& uid, int offset, int count) NN_NOEXCEPT;

    /*!
        @brief      ブロックリストを取得します。

        @param[out] outCount    取得した数。
        @param[out] outUsers    ブロックリスト。
        @param[in]  uid         ユーザーアカウント。
        @param[in]  offset      オフセット。
        @param[in]  count       ブロックリストの要素数。

        @return     処理結果。

        @pre
            - outCount != nullptr
            - outUsers != nullptr
    */
    nn::Result GetBlockedUserList(int* outCount, BlockedUserImpl* outUsers,
        const nn::account::Uid& uid, int offset, int count) NN_NOEXCEPT;

    /*!
        @brief      ブロックリストを更新します。

        @param[in]  uid         ユーザーアカウント。
        @param[in]  resources   ブロックしたユーザーリソースリスト。
        @param[in]  count       ブロックしたユーザーリソースリストの要素数。

        @return     処理結果。

        @pre
            - resources != nullptr
    */
    nn::Result UpdateBlockedUserList(const nn::account::Uid& uid, const BlockedUserResource* resources, int count) NN_NOEXCEPT;

    /*!
        @brief      ユーザーを削除します。

        @param[in]  uid         ユーザーアカウント。
        @param[in]  accountId   ブロックしたユーザーのネットワークサービスアカウント ID。

        @return     処理結果。
    */
    nn::Result DeleteUser(const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT;

    /*!
        @brief      メモリキャッシュを無効化します。
    */
    void Invalidate() NN_NOEXCEPT;

private:
    //
    nn::os::Mutex m_Mutex;
    //
    nn::account::Uid m_CurrentUid;
    //
    Record m_Records[BlockedUserCountMax];
    int m_Count;

private:
    //
    nn::Result Load(const nn::account::Uid& uid) NN_NOEXCEPT;
    nn::Result LoadImpl(const nn::account::Uid& uid) NN_NOEXCEPT;
    //
    nn::Result Save() NN_NOEXCEPT;
    //
    int SearchRecord(nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT;
};

}}}}}
