﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/friends/friends_Types.h>

namespace nn { namespace friends { namespace detail {

/*!
    @brief      プレゼンスデータのアクセサーです。
*/
class PresenceAccessor
{
public:
    /*!
        @brief      キーバリューのポインタを格納する構造体です。
    */
    struct KeyValue
    {
        const char* key;
        const char* value;
    };

public:
    /*!
        @brief      キーバリューを追加します。

        @param[in]  key     キー。
        @param[in]  value   バリュー。
        @param[in]  buffer  バッファ。
        @param[in]  size    バッファサイズ。

        @return     処理結果。

        @pre
            - key != nullptr
            - value != nullptr
            - buffer != nullptr
            - size > 0

        @details
                    指定したバッファにキーバリューを書き込みます。@n
                    バッファの不足やキーの重複を検知した場合、エラーが返ります。
    */
    static nn::Result AddKeyValue(const char* key, const char* value, char* buffer, size_t size) NN_NOEXCEPT;

    /*!
        @brief      キーの検索を行います。

        @param[in]  key     キー。
        @param[in]  buffer  バッファ。
        @param[in]  size    バッファサイズ。

        @return     バリュー。

        @pre
            - key != nullptr
            - buffer != nullptr
            - size > 0

        @details
                    キーが存在しない場合、nullptr が返ります。
    */
    static const char* SearchKey(const char* key, const char* buffer, size_t size) NN_NOEXCEPT;

    /*!
        @brief      キーバリューのリストを取得します。

        @param[out] outCount        取得したキーバリューの数。
        @param[out] outKeyValues    キーバリューリスト。
        @param[in]  count           キーバリューリストの要素数。
        @param[in]  buffer          バッファ。
        @param[in]  size            バッファサイズ。

        @return     処理結果。

        @pre
            - outCount != nullptr
            - outKeyValues != nullptr
            - count > 0
            - buffer != nullptr
            - size > 0
    */
    static nn::Result GetKeyValueList(int* outCount, KeyValue* outKeyValues, int count, const char* buffer, size_t size) NN_NOEXCEPT;

    /*!
        @brief      キーを検証します。

        @param[in]  key     キー。
        @param[in]  length  キーの長さ。

        @return     キーが正しいかどうか。

        @pre
            - key != nullptr
    */
    static bool VerifyKey(const char* key, size_t length) NN_NOEXCEPT;

    /*!
        @brief      バリューを検証します。

        @param[in]  value   バリュー。
        @param[in]  length  バリューの長さ。

        @return     バリューが正しいかどうか。

        @pre
            - value != nullptr
    */
    static bool VerifyValue(const char* value, size_t length) NN_NOEXCEPT;
};

}}}
