﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/friends/friends_Types.h>
#include <nn/friends/friends_AsyncContext.h>
#include <nn/friends/detail/ipc/friends_IFriendService.sfdl.h>
#include <nn/os.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/nn_StaticAssert.h>

namespace nn { namespace friends { namespace detail {

/*!
    @brief      非同期処理を扱うクラスです。
*/
class AsyncContextInternal
{
public:
    /*!
        @brief      非同期呼び出しを行うスレッドのスタックサイズです。
    */
    static const size_t ThreadStackSize = 12 * 1024;
    /*!
        @brief      AsyncContextInternal オブジェクトに割り当てるバッファのサイズです。
    */
    static const size_t ObjectBufferSize = 2 * 1024;
    /*!
        @brief      オブジェクトが確保するバッファサイズです。
    */
    static const size_t AllocateBufferSize = ThreadStackSize + ObjectBufferSize * 2;

    /*!
        @brief      AsyncContext が呼び出す同期関数です。

        @details
                    非同期呼び出ししたい関数の第一引数は、必ず std::move(cancelHandle) を指定してください。@n
                    cancelHandle の実態は、 AsyncContextInternal が保持する SystemEvent の ReadableHandle です。

                    処理結果を return することで、 AsyncContext::GetResult で取得する値が設定されます。
    */
    typedef nn::Result (*SyncCall)(detail::ipc::IFriendService* session, void* param);

    /*!
        @brief      パラメータのディープコピーを行うコールバック関数です。
    */
    typedef void (*DeepCopyCallback)(const void* source, void* dest, size_t destSize);

public:
    /*!
        @brief      AsyncContextInternal インスタンスを作成します。

        @param[out] outInstance         インスタンス。
        @param[in]  session             セッション。
        @param[in]  allocateFunction    メモリ確保関数。
        @param[in]  deallocateFunction  メモリ解放関数。

        @return     処理結果。
    */
    static nn::Result CreateInstance(AsyncContextInternal** outInstance,
        detail::ipc::IFriendService* session, nn::AllocateFunction allocateFunction, nn::DeallocateFunction freeFunction) NN_NOEXCEPT;

    /*!
        @brief      AsyncContextInternal インスタンスを破棄します。

        @param[out] instance    インスタンス。

        @return     処理結果。
    */
    static void DeleteInstance(AsyncContextInternal* instance) NN_NOEXCEPT;

public:
    /*!
        @brief      非同期呼び出しを行います。

        @param[in]  threadName  スレッド名。
        @param[in]  function    関数。
        @param[in]  param       関数に渡すパラメータ。
        @param[in]  paramSize   関数に渡すパラメータのサイズ。
    */
    void CallAsync(const char* threadName,
        SyncCall function, void* param, size_t paramSize) NN_NOEXCEPT;

    /*!
        @brief      非同期呼び出しを行います。

        @param[in]  threadName  スレッド名。
        @param[in]  function    関数。
        @param[in]  callback    ディープコピーコールバック。
        @param[in]  sourceParam ディープコピーコールバックに渡すソースパラメータ。
    */
    void CallAsync(const char* threadName,
        SyncCall function, DeepCopyCallback callback, const void* sourceParam) NN_NOEXCEPT;

    /*!
        @brief      処理を中断します。
    */
    void Cancel() NN_NOEXCEPT;

    /*!
        @brief      処理が完了したかどうかを確認します。

        @return     処理が完了したかどうか。
    */
    bool IsCompleted() NN_NOEXCEPT;

    /*!
        @brief      処理結果を取得します。

        @return     処理結果。
    */
    nn::Result GetResult() const NN_NOEXCEPT;

    /*!
        @brief      非同期処理完了イベントの ReadableHandle を取り付けます。

        @param[out] outEvent    イベント。
    */
    void AttachCompletionEventReadableHandle(nn::os::SystemEvent *outEvent) NN_NOEXCEPT;

private:
    //
    mutable nn::os::SdkMutexType m_Mutex;
    //
    nn::os::ThreadType m_Thread;
    bool m_IsThreadCreated;
    //
    nn::os::SystemEventType m_CompletionEvent;
    bool m_IsCompletionEventCreated;
    //
    SyncCall m_Function;
    Bit64 m_Param[1024 / sizeof (Bit64)];
    nn::Result m_Result;
    //
    detail::ipc::IFriendService* m_Session;
    //
    void* m_Stack;
    void* m_AllocatedBuffer;
    //
    nn::DeallocateFunction m_FreeFunction;

private:
    /*!
        @brief      コンストラクタです。
    */
    AsyncContextInternal() NN_NOEXCEPT;

    /*!
        @brief      デストラクタです。
    */
    ~AsyncContextInternal() NN_NOEXCEPT;

    //
    static void FunctionCaller(void* arg) NN_NOEXCEPT;
};

NN_STATIC_ASSERT(sizeof (AsyncContextInternal) < AsyncContextInternal::ObjectBufferSize - sizeof (std::max_align_t));

}}}
