﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*
    @file
    @brief  Public header defining FGM debugger API.
 */

#pragma once

#include <nn/os/os_SystemEvent.h>
#include <nn/nn_TimeSpan.h>
#include <nn/fgm/fgm_Types.h>
#include <nn/fgm/sfdl/fgm.sfdl.h>
#include <nn/fgm/fgm_DebugTypes.h>

namespace nn  {
namespace fgm {

/**
    @brief      Implements FGM logger.

    @details    FGM is equipped with facility to collect log of events. The log will
                contains information about each request including timestamp, min/max,
                pid of the caller and the actual state of the PM resource.

                FGM log is set up to use memory provided by each debug client. Size of
                internal log is thus determined by the size of the memory the client
                provides. Log entries are stored in the circular event log. The overruns
                may happen if for example client is slow to retrieve log entries, in such
                scenario older entries are discarded.

                Whenever a new entry is appended to the log, event is signaled. This event
                can be used by the client to schedule retrieval of log entries.

                Sample code:
                @code
                #include <nn/fgm_Debugger.h>
                nn::fgm::Debugger debugger;
                uint8_t logMemory[16 * 1024] NN_ALIGNAS(4096);
                void test()
                {
                    nn::Result result;
                    nn::fgm::DebugEntry entries[4];
                    result = debugger.Initialize(logMemory, sizeof(logMemory));
                    if (result.IsSuccess())
                    {
                        while (debugger.Wait().IsSuccess())
                        {
                            do
                            {
                                result = debugger.Read(entries, sizeof(entries)/sizeof(entries[0]), &read, &lost, &remaining);
                            }
                            while (result.IsSuccess() && remaining);
                        }
                    }
                }
                @endcode
*/
class Debugger
{
    NN_DISALLOW_COPY(Debugger);
    NN_DISALLOW_MOVE(Debugger);

private:
    nn::sf::SharedPointer<nn::fgm::sf::IDebugger> m_DebuggerImpl;
    nn::os::SystemEvent m_SystemEvent;

public:
    /**
        @brief      Initialize debugger.
        @param[in]  pMemoryPool             Memory for log collection.
        @param[in]  memoryPoolSize          Size of the memory for log collection.
        @param[in]  eventClearMode          Clear mode for log update event.
        @return     nn::ResultSuccess()     Operation completed successfully.
        @return     nn::Result              Failure code.

        @details    The Initialize() function is used to initialize debugger instance.
                    In order to use FGM debug facility, client needs to be provisioned
                    to use FGM logger. Provisioning is done by adding FGM debugger
                    port name specified in fgm_Types.h to client's metadata.
    */
    nn::Result Initialize(void* pMemoryPool, size_t memoryPoolSize, nn::os::EventClearMode eventClearMode) NN_NOEXCEPT;

    /**
        @brief      Initialize debugger instance.
        @param[in]  pMemoryPool             Memory for log collection.
        @param[in]  memoryPoolSize          Size of the memory for log collection.
        @return     nn::ResultSuccess()     Operation completed successfully.
        @return     nn::Result              Failure code.

        @details    This function call is similar to @ref Initialize. However it sets up
                    event clear mode to manual.
    */
    nn::Result Initialize(void* pMemoryPool, size_t memoryPoolSize) NN_NOEXCEPT;

    /**
        @brief      Wait for log update notification.
        @return     nn::ResultSuccess()     Operation completed successfully.
        @return     nn::Result              Failure code.

        @details    The Wait() function may be used to block waiting for log update
                    notification. Before returning this function will manually clear event.
    */
    nn::Result Wait() NN_NOEXCEPT;

    /**
        @brief      Wait for log update notification with timeout.
        @return     nn::ResultSuccess()     Operation completed successfully.
        @return     nn::Result              Failure code.

        @details    The WaitWithTimeout() function may be used to block with timeout
                    waiting for log update notifications. Before returning, this function will
                    manually clear event.
    */
    nn::Result WaitWithTimeout(nn::TimeSpan timeout) NN_NOEXCEPT;

    /**
        @brief      Obtain handle to system event associated with this debugger.
        @return     Pointer to system event.

        @details    Clients wishing to implement their own handler for log update
                    notifications may obtain a pointer to the system event associated
                    with this debugger instance. System event by default is
                    using manual clear mode. If auto clear mode is desired, please
                    use @ref Initialize() function version that accepts argument
                    for event clear mode.
    */
    nn::os::SystemEvent* GetSystemEventPointer() NN_NOEXCEPT;

    /**
        @brief      Manually clear event.
        @details    Can be used to manually clear event.
    */
    void       ClearEvent() NN_NOEXCEPT;

    /**
        @brief      Cancel operation.
        @return     nn::ResultSuccess()     Operation completed successfully.
        @return     nn::Result              Failure code.

        @details    The Cancel() function may be used to abort pending operations.
                    This call will change state of the event to 'signaled'.
                    This call may be used to unblock @ref Wait() or
                    @ref WaitWithTimeout().
    */
    nn::Result Cancel() NN_NOEXCEPT;

    /**
        @brief      Read log entries
        @param[in]  pEntriesOut             Buffer for log entries.
        @param[in]  maxCountToRead          Number of entries to read.
        @param[in]  pReadOut                Number of entries read.
        @param[in]  pLostOut                Number of entries lost due to log overrun since last call.
        @param[in]  pRemainingOut           Number of entries still in the log.
        @return     nn::ResultSuccess()     Operation completed successfully.
        @return     nn::Result              Failure code.

        @details    The Read() function is used to read log entries. Log entries are returned in
                    the buffer specified with @a pEntriesOut. Maximum number of entries to read
                    is specified with @a maxCountToRead.

                    On return function fill in:
                    pReadOut - number of actual entries returned in @a pEntries
                    pLostOut - number of overruns since last call to Read
                    pRemeiningOut - number of unread entries remaining in the log
    */
    nn::Result Read(DebugEntry* pEntriesOut, uint32_t maxCountToRead, uint32_t* pReadOut, uint32_t* pLostOut, uint32_t* pRemainingOut) NN_NOEXCEPT;

    /**
        @brief      Deinitialize debugger instance.
        @return     nn::ResultSuccess()     Operation completed successfully.
        @return     nn::Result              Failure code.

        @details    The Finalize() call is used to de-associated this debugger from FGM.
                    Once called, memory provided by the client is released by FGM.
    */
    nn::Result Finalize() NN_NOEXCEPT;

    Debugger() NN_NOEXCEPT;
    ~Debugger() NN_NOEXCEPT;
};

}}
