﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>
#include <nn/fgm/fgm_Types.h>

namespace nn     {
namespace fgm    {

/**
    @brief      Event id
    @details    FGM log entries will contain one of the following event id's.
*/
enum EventId
{
    /**
        @brief New min-max request added.
    */
    EventId_Set,

    /**
        @brief Previous min-max request removed.
    */
    EventId_Unset,

    /**
        @brief Hardware handler PM resource completed transition.
    */
    EventId_Completion
};

/**
    @brief      Contains information for added or removed request.
    @details
*/
struct RequestEntry
{
    /**
        @brief  Pid of request owner.
    */
    os::ProcessId pid;

    /**
        @brief  Priority of the request.
    */
    Priority priority;

    /**
        @brief  Requested minimum.
    */
    Setting  min;

    /**
        @brief  Requested maximum.
    */
    Setting  max;
};

/**
    @brief      Contains information about request sent to hardware handler.
    @details
*/
struct CompletionEntry
{
    /**
        @brief  Requested minimum.
    */
    Setting  min;

    /**
        @brief  Requested maximum.
    */
    Setting  max;

    /**
        @brief  Actual after operation completed.
    */
    Setting  actual;
};

/**
    @brief      FGM log entry.
    @details
*/
struct DebugEntry
{
    /**
        @brief Event id: EventId_Set, EventId_Unset, EventId_Completion
    */
    EventId  event;

    /**
        @brief Resource module id
    */
    Module   id;

    /**
        @brief Timestamp based on system tick
        @see   nn::os::GetSystemTick()
    */
    nn::os::Tick timestamp;

    union
    {
        /**
            @brief Event information, used for EventId_Set, EventId_Unset
        */
        RequestEntry    request;

        /**
            @brief Event information, used for EventId_Completion
        */
        CompletionEntry completion;

        uint64_t        reserved[8];
    } u;

};

}}
