﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/util/util_BitFlagSet.h>
#include <string>

namespace nn { namespace fatalsrv {

#define NN_FATALSRV_AARCH32_VALUES(ENTRY) \
    ENTRY(R0,    0)\
    ENTRY(R1,    1)\
    ENTRY(R2,    2)\
    ENTRY(R3,    3)\
    ENTRY(R4,    4)\
    ENTRY(R5,    5)\
    ENTRY(R6,    6)\
    ENTRY(R7,    7)\
    ENTRY(R8,    8)\
    ENTRY(R9,    9)\
    ENTRY(R10,  10)\
    ENTRY(R11,  11)\
    ENTRY(R12,  12)\
    ENTRY(R13,  13)\
    ENTRY(R14,  14)\
    ENTRY(R15,  15)\
    ENTRY(Pstate,   16)\
    ENTRY(Afsr0,    17)\
    ENTRY(Afsr1,    18)\
    ENTRY(Esr,      19)\
    ENTRY(Far,      20)\


#define NN_FATALSRV_GENERATE_ENUM(NAME, NUMBER) NAME=NUMBER,
    struct Aarch32Context
    {
        static const int MaxBacktraceCount = 32;

        union
        {
            struct
            {
                // プロセッサ汎用レジスタ
                union
                {
                    struct
                    {
                        Bit32   r[16];      //!< r0～r15 の値が格納されます
                    };
                    struct
                    {
                        Bit32   _padding[9];
                        Bit32   sb;         //!< r[ 9] と等価です
                        Bit32   sl;         //!< r[10] と等価です
                        Bit32   fp;         //!< r[11] と等価です
                        Bit32   ip;         //!< r[12] と等価です
                        Bit32   sp;         //!< r[13] と等価です。スタックポインタです
                        Bit32   lr;         //!< r[14] と等価です。リンクレジスタです
                        Bit32   pc;         //!< r[15] と等価です。プログラムカウンタです
                    };
                };

                // 例外情報
                Bit32   pstate; //!< pstate の値が格納されます。ビット配置は SPSR のフォーマットと同じです
                Bit32   afsr0;  //!< AFSR0 の値が格納されます。
                Bit32   afsr1;  //!< AFSR1 の値が格納されます。
                Bit32   esr;    //!< ESR の値が格納されます。
                Bit32   far;    //!< FAR の値が格納されます。
            };
            struct
            {
                // 汎用レジスタと例外情報をまとめてアクセスするための配列
                Bit32   values[21];
            };
        };
                                    // バックトレース
        Bit32   backtrace[MaxBacktraceCount];
        int32_t backtraceCount;

        // プログラムのマップされているアドレス
        Bit32   programMappedAddr;

        // 各項目のセット/非セットを表す
        nn::util::BitFlagSet<32> setFlag;

        enum ValueNames
        {
            NN_FATALSRV_AARCH32_VALUES(NN_FATALSRV_GENERATE_ENUM)
            ValueNameCount,
        };

        static const char* ValueNameStrings[];

        void Clear() NN_NOEXCEPT
        {
            std::memset(this, 0, sizeof(*this));
        }

        void SetValue(ValueNames target, Bit32 value) NN_NOEXCEPT
        {
            this->values[target] = value;
            setFlag.Set(target, true);
        }

        void SetProgramMappedAddr(Bit32 addr) NN_NOEXCEPT
        {
            this->programMappedAddr = addr;
        }
    };
}} // namespace nn::fatalsrv
