﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
*   @file
*   @brief  eTicketService の権利判定に関する型や定数の宣言
*/

#pragma once

#include <nn/es/es_Types.h>
#include <nn/es/es_ELicenseTypes.h>
#include <nn/account/account_NintendoAccountTypes.h>
#include <nn/util/util_BitFlagSet.h>

namespace nn { namespace es {

struct RightsStatusFlagTag {};
typedef util::BitFlagSet<32, RightsStatusFlagTag> RightsStatusFlag;

typedef RightsStatusFlag::Flag<0>   RightsStatusFlag_Permanent;                     //!< 利用期限がない権利
typedef RightsStatusFlag::Flag<1>   RightsStatusFlag_RecommendsServerInteraction;   //!< 起動時にネットワークでの再認証を推奨する権利


/*!
*   @brief      権利の情報を表す構造体です。
*/
struct Rights
{
    /*!
    *   @brief      権利が利用できる場合の権利の参照元 ELicenseId です。
    *   @details    コモンチケットや ELicense により権利が管理されていない場合のパーソナライズドチケットにより権利を保持している場合は
    *               ELicenseId::InvalidId が入ります。
    */
    ELicenseId eLicenseId;

    /*!
    *   @brief      eLicenseId が指す対象 eLicense の利用期限です。
    */
    ExpireDate expireDate;

    // eLicenseId が指す対象 eLicense の所有者 ID の即値です。
    uint64_t _eLicenseOwnerIdValue;

    /*!
    *   @brief      eLicenseId が指す対象 eLicense の所有者 NAID を返します。
    */
    account::NintendoAccountId GetOwnerNintendoAccountId() const NN_NOEXCEPT
    {
        return { _eLicenseOwnerIdValue };
    }

    /*!
    *   @brief      権利の RightsId です。
    */
    RightsId rightsId;

    /*!
    *   @brief      権利の属性に関する情報を格納するフラグです。
    */
    RightsStatusFlag _flags;

    /*!
    *   @brief      拡張のための予約領域です。
    */
    uint8_t _reserved[4];

    /*!
    *   @brief      利用期限がない権利であるかを判定します。
    */
    bool IsPermanentRights() const NN_NOEXCEPT
    {
        return _flags.Test(RightsStatusFlag_Permanent::Index);
    }

    /*!
    *   @brief      起動時にネットワークでの再認証を推奨する権利かを判定します。
    */
    bool IsRecommendsServerInteraction() const NN_NOEXCEPT
    {
        return _flags.Test(RightsStatusFlag_RecommendsServerInteraction::Index);
    }

    bool operator==(const Rights& rhs) const NN_NOEXCEPT
    {
        return rightsId == rhs.rightsId
            && eLicenseId == rhs.eLicenseId
            && _flags._storage[0] == rhs._flags._storage[0];
    }

    bool operator!=(const Rights& rhs) const NN_NOEXCEPT
    {
        return !(*this == rhs);
    }
};

/*!
*   @brief 権利の状態を表す構造体です。
*/
struct RightsStatus
{
    /*!
    *   @brief      権利の情報です。
    */
    Rights rights;

    /*!
    *   @brief      権利の状態です。
    */
    enum Status : uint32_t
    {
         Available,                         //!< アカウントに限らず利用可能です。
         AvailableSinceAccountRestricted,   //!< 現在登録されているアカウントでは利用可能です。
         Prepurchased,                      //!< いずれかのアカウントで予約されています。
         HasAccountRestrictedRights,        //!< 利用不可能ですが、アカウントを限定すれば利用可能です。
         NotAvailable,                      //!< 利用不可能です。
    } status;

    /*!
    *   @brief      拡張のための予約領域です。
    */
    uint8_t _reserved[4];
};

}}
