﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
*   @file
*   @brief  eTicketService のローカルでの同時利用判定に関する API
*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/es/es_LocalConcurrencyCheckTypes.h>

//--------------------------------------------------------------------------
//  C++ 向けの宣言
//--------------------------------------------------------------------------
namespace nn { namespace es {

/**
*   @brief      ローカルでの同時利用判定のためのチャレンジを取得します。
*
*   @param[out] outValue                                                        取得したチャレンジ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                                                   成功しました。
*
*   @details    この関数はローカル通信での同時利用判定に使われます。
*               取得したチャレンジは GetLocalConcurrencyCheckData() で使用してください。
*/
Result GetChallengeOfLocalConcurrencyCheck(LocalConcurrencyCheckChallenge* outValue) NN_NOEXCEPT;

/**
*   @brief      ローカルでの同時利用判定を行います。
*
*   @param[out] outInfo                                                         取得した認証情報
*   @param[in]  authenticatedDataList                                           既に認証済みの権利利用情報の配列
*   @param[in]  authenticatedDataListCount                                      既に認証済みの権利利用情報の配列の要素数
*   @param[in]  unauthenticatedData                                             認証を行う権利利用情報
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                                                   成功しました。
*   @retval     ResultLocalConcurrencyCheckIllegalConcurrencyDetected           重複した権利の利用がありました。
*   @retval     ResultLocalConcurrencyCheckDataVerificationFailed               権利利用情報の検証に失敗しました。
*
*   @details    この関数はローカル通信での同時利用判定に使われます。
*               同時利用判定を行う全てデバイス(自デバイスまたは他デバイス)で GetLocalConcurrencyCheckData() で取得した権利利用情報をこの関数に渡してください。
*               取得した認証情報は VerifyLocalConcurrencyAuthentication() で検証してください。
*/
Result AuthorizeLocalConcurrencyCheckData(LocalConcurrencyCheckAuthenticationInfo* outInfo, const LocalConcurrencyCheckData authenticatedDataList[], int authenticatedDataListCount, const LocalConcurrencyCheckData& unauthenticatedData) NN_NOEXCEPT;

/**
*   @brief      ローカルでの同時利用判定のための権利利用情報を取得します。
*
*   @param[out] outData                                                         取得した権利利用情報
*   @param[in]  challenge                                                       発行されたチャレンジ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                                                   成功しました。
*
*   @details    この関数はローカル通信での同時利用判定に使われます。
*               取得した権利利用情報は AuthorizeLocalConcurrencyCheckData() で検証してください。
*/
Result GetLocalConcurrencyCheckData(LocalConcurrencyCheckData* outData, const LocalConcurrencyCheckChallenge& challenge) NN_NOEXCEPT;

/**
*   @brief      ローカルでの同時利用判定のための認証情報の検証を行います。
*
*   @param[in]  info                                                            検証を行う認証情報
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                                                   成功しました。
*   @retval     ResultLocalConcurrencyCheckAuthenticationInfoVerificationFailed 認証情報の検証に失敗しました。
*
*   @details    この関数はローカル通信での同時利用判定に使われます。
*/
Result VerifyLocalConcurrencyAuthentication(const LocalConcurrencyCheckAuthenticationInfo& info) NN_NOEXCEPT;
}}

