﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
*   @file
*   @brief  eTicketService の eLicense に関する型や定数の宣言
*/

#pragma once

#include <cctype>
#include <cstdlib>
#include <limits>
#include <nn/nn_SdkAssert.h>
#include <nn/account/account_NintendoAccountTypes.h>
#include <nn/es/es_Types.h>
#include <nn/util/util_BitFlagSet.h>
#include <nn/util/util_TFormatString.h>

namespace nn { namespace es {

/**
*   @brief 128bit の Id を表す構造体です。
*/
struct Bit128IdBase
{
    static const int StringSize = 32;

    uint8_t _data[16];

    friend bool operator ==(const Bit128IdBase& lhs, const Bit128IdBase& rhs) NN_NOEXCEPT
    {
        return std::memcmp(lhs._data, rhs._data, sizeof(lhs)) == 0;
    }

    friend bool operator !=(const Bit128IdBase& lhs, const Bit128IdBase& rhs) NN_NOEXCEPT
    {
        return !(lhs == rhs);
    }

    /**
    * @brief   16 進数 32 文字の文字列表現を取得します。
    *
    * @param[out]  pOutString  文字列を代入するバッファへのポインタを指定します。
    * @param[in]   size        pOutString に渡したバッファのサイズを指定します。
    *
    * @return  pOutString を返します。
    *
    * @pre
    *  - pOutString != nullptr
    *  - size >= StringSize + 1
    *
    * @details
    *  pOutString で渡されたバッファに文字列表現を書き込みます。
    */
    const char* ToString(char* pOutString, size_t size) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(size >= StringSize + 1);

        util::TSNPrintf(
            pOutString, size, "%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x",
            _data[0], _data[1], _data[2], _data[3], _data[4], _data[5], _data[6], _data[7],
            _data[8], _data[9], _data[10], _data[11], _data[12], _data[13], _data[14], _data[15]
        );

        return pOutString;
    }

    /**
    * @brief   16 進数 32 文字の文字列表現から値を取得します。
    *
    * @param[in]   pInString   変換する文字列表現へのポインタを指定します
    *
    * @return  変換が成功したかを返します。
    *
    * @pre
    *  - pInString != nullptr
    */
    bool FromString(const char* pInString) NN_NOEXCEPT
    {
        for (int i = 0; i < 16; ++i, pInString += 2)
        {
            if (!(std::isxdigit(static_cast<unsigned char>(pInString[0])) && std::isxdigit(static_cast<unsigned char>(pInString[1]))))
            {
                return false;
            }

            char buf[3] = { pInString[0], pInString[1], '\0' };
            _data[i] = static_cast<uint8_t>(strtoul(buf, nullptr, 16));
        }

        return pInString[0] == '\0';
    }
};

/**
*   @brief ELicenseIdを表す数値です。
*/
typedef Bit128IdBase ELicenseId;

/**
*   @brief 無効なELicenseIdです。
*/
const ELicenseId InvalidELicenseId =
{
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
};

/**
*   @brief ELicenseArchiveIdを表す数値です。
*/
typedef Bit128IdBase ELicenseArchiveId;

/**
*   @brief 無期限を表す定数です。
*/
const int64_t ExpireDateValueAsPermanent = std::numeric_limits<int64_t>::max();

/**
*   @brief 利用期限を表す構造体です。
*/
struct ExpireDate
{
    /**
    *   @brief 利用期限の時間です。
    */
    time::PosixTime time;

    /**
    *   @brief 利用期限が無期限であるかを判定します。
    */
    bool IsPermanent() const NN_NOEXCEPT
    {
        return time.value == ExpireDateValueAsPermanent;
    }
};

/**
*   @brief ELicense の権利の利用範囲を表す列挙体です。
*/
enum class ELicenseScope : uint8_t
{
    Account,    //!< アカウント
    Device,     //!< デバイス
};

/**
*   @brief ELicense の情報を表す構造体です。
*/
struct ELicenseInfo
{
    /**
    *   @brief ELicenseIdです。
    */
    ELicenseId eLicenseId;

    /**
    *   @brief 対応するチケットのRightsIdです。
    */
    RightsId rightsId;

    /**
    *   @brief 対応するチケットのTicketIdです。
    */
    TicketId ticketId;

    /**
    *   @brief 所有者のNintendoAccountIdです。
    */
    account::NintendoAccountId ownerNaId;

    /**
    *   @brief 利用期限です。
    */
    ExpireDate expireDate;

    /**
    *   @brief 対応するチケットの所有者のAccountIdです。
    */
    AccountId ticketOwnerVaId;

    /**
    *   @brief 利用範囲です。
    */
    ELicenseScope scope;

    /**
    *   @brief 将来の拡張のための予約領域です。
    */
    uint8_t padding[11];
};
NN_STATIC_ASSERT(sizeof(ELicenseInfo) == 64);

struct ELicenseStatusFlagTag {};
typedef util::BitFlagSet<32, ELicenseStatusFlagTag> ELicenseStatusFlag;

typedef ELicenseStatusFlag::Flag<0>   ELicenseStatusFlag_HasTicket; //!< 対応するチケットを保有している

/**
*   @brief ELicense と付随する情報を含んだ構造体です。
*/
struct ELicenseInfoWrapper
{
    /**
    *   @brief ELicense の情報です。
    */
    ELicenseInfo info;

    /**
    *   @brief 付随する情報を格納するフラグです。
    */
    ELicenseStatusFlag _flags;

    /**
    *   @brief 拡張のための予約領域です。
    */
    uint8_t _reserved[4];

    /**
    *   @brief 対応するチケットを保有しているかを判定します。
    */
    bool HasTicket() const NN_NOEXCEPT
    {
        return _flags.Test(ELicenseStatusFlag_HasTicket::Index);
    }
};
NN_STATIC_ASSERT(sizeof(ELicenseInfoWrapper) == 72);

}}
