﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
*   @file
*   @brief  eTicketService の eLicense に関する API
*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/account/account_NintendoAccountTypes.h>
#include <nn/es/es_ELicenseTypes.h>

//--------------------------------------------------------------------------
//  C++ 向けの宣言
//--------------------------------------------------------------------------
namespace nn { namespace es {

/**
*   @brief      eLicense アーカイブのインポートコンテキストです。
*/
struct ELicenseImportContext
{
    uint64_t _data[2];

    /**
    *   @brief      Challenge を表す数値です。
    *   @details    サーバに値を送信するときは、value の値を16進数16文字の文字列に変換してください。
    */
    struct Challenge
    {
        uint64_t value;

        friend bool operator==(const Challenge& lhs, const Challenge& rhs) NN_NOEXCEPT
        {
            return lhs.value == rhs.value;
        }

        friend bool operator!=(const Challenge& lhs, const Challenge& rhs) NN_NOEXCEPT
        {
            return !(lhs == rhs);
        }
    };

    /**
    *   @brief      eLicense アーカイブの発行用のチャレンジを取得します。
    *   @return     アーカイブの発行用のチャレンジ
    */
    Challenge GetChallenge() NN_NOEXCEPT;
};

/**
*   @brief      eLicense アーカイブのインポートを開始します。
*
*   @param[out] outContext                  取得したコンテキスト
*   @param[in]  naId                        インポートする eLicense アーカイブの NintendoAccountId
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*/
Result BeginImportELicenseArchive(ELicenseImportContext* outContext, account::NintendoAccountId naId) NN_NOEXCEPT;

/**
*   @brief      eLicense アーカイブのファイルをインポートします。
*               分割して eLicense アーカイブをインポートすることも可能です。
*               その場合でもファイルの順番を変えずに順次インポートしてください。
*
*   @param[in]  context                     コンテキスト
*   @param[in]  eLicenseArchive             インポートする eLicense アーカイブを格納するバッファ
*   @param[in]  eLicenseArchiveSize         インポートする eLicense アーカイブを格納するバッファのサイズ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*/
Result ImportELicenseArchive(const ELicenseImportContext& context, void* eLicenseArchive, size_t eLicenseArchiveSize) NN_NOEXCEPT;

/**
*   @brief      eLicense アーカイブのインポートを完了します。
*
*   @param[out] outELicenseArchiveId        取得した ELicenseArchiveId
*   @param[in]  context                     コンテキスト
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*/
Result EndImportELicenseArchive(ELicenseArchiveId* outELicenseArchiveId, const ELicenseImportContext& context) NN_NOEXCEPT;

/**
*   @brief      有効な eLicense の数を取得します。
*
*   @return     取得した eLicense の数
*/
int CountELicense() NN_NOEXCEPT;

/**
*   @brief      有効な eLicense の ELicenseId のリストを取得します。
*
*   @param[out] outELicenseIdList[]         取得した ELicenseId を格納する配列
*   @param[in]  count                       取得した ELicenseId を格納する配列の要素数
*
*   @return     取得した eLicense の ELicenseId の数
*/
int ListELicenseIds(ELicenseId outELicenseIdList[], int count) NN_NOEXCEPT;

/**
*   @brief      eLicense の情報を取得します。
*
*   @param[out] outELicenseInfoList[]       取得した eLicense の情報を格納する配列
*   @param[in]  eLicenseInfoCount           取得した eLicense の情報を格納する配列の要素数
*   @param[in]  eLicenseIdList[]            eLicense の情報を取得する ELicenseId の配列
*   @param[in]  eLicenseIdCount             eLicense の情報を取得する ELicenseId の配列の要素数
*
*   @return     eLicense の情報を取得した数
*
*   @details    入力された ELicenseId のなかで存在している eLicense の情報を取得します。
*               存在しなかった eLicense は戻り値の数にカウントされません。
*/
int ListELicenseInfo(ELicenseInfoWrapper outELicenseInfoList[], int eLicenseInfoCount, const ELicenseId eLicenseIdList[], int eLicenseIdCount) NN_NOEXCEPT;

}}

