﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
*   @file
*   @brief  eTicketService に関する API
*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include <nn/es/es_Types.h>

//--------------------------------------------------------------------------
//  C++ 向けの宣言
//--------------------------------------------------------------------------
namespace nn { namespace es {

/**
*   @brief      チケットのインポートを行います。
*
*   @param[in]  ticket                      インポートするチケットのポインタ
*   @param[in]  ticketSize                  インポートするチケットのサイズ
*   @param[in]  certificate                 インポートする証明書のポインタ
*   @param[in]  certificateSize             インポートする証明書のサイズ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*   @retval     ResultTicketInvalid         チケットが不正です。
*   @retval     ResultCertificateInvalid    証明書が不正です。
*   @retval     ResultTicketFull            チケットの上限数を超えています。
*   @retval     ResultTicketVersionInvalid  現在保存されているチケットより古いバージョンのチケットです。
*/
Result ImportTicket(const void* ticket, size_t ticketSize, const void* certificate, size_t certificateSize) NN_NOEXCEPT;

/**
*   @brief      チケットのインポートを行います。
*
*   @param[in]  ticketCertificateSet        インポートするチケットと証明書のポインタ
*   @param[in]  ticketCertificateSetSize    インポートするチケットと証明書のサイズ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*   @retval     ResultTicketInvalid         チケットが不正です。
*   @retval     ResultCertificateInvalid    証明書が不正です。
*   @retval     ResultTicketFull            チケットの上限数を超えています。
*/
Result ImportTicketCertificateSet(const void* ticketCertificateSet, size_t ticketCertificateSetSize) NN_NOEXCEPT;

/**
*   @brief      予約購入記録のインポートを行います。
*
*   @param[in]  record                      インポートする予約購入記録
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*   @retval     ResultTicketFull            チケットの上限数を超えています。
*/
Result ImportPrepurchaseRecord(const nn::es::PrepurchaseRecord& record) NN_NOEXCEPT;

/**
*   @brief      RightsID でチケットの複数削除を行います。
*
*   @param[in]  rightsIdList[]  削除するチケットの 128bit RightsId の配列
*   @param[in]  count           削除するチケットの要素数
*/
void DeleteTicket(const nn::es::RightsIdIncludingKeyId rightsIdList[], int count) NN_NOEXCEPT;

/**
*   @brief      AccountId が一致する全ての Personalized チケットの削除を行います。
*
*   @param[in]  accountId   削除するチケットの AccountId
*/
void DeletePersonalizedTicket(nn::es::AccountId accountId) NN_NOEXCEPT;

/**
*   @brief      指定した予約購入記録の削除を行います。
*
*   @param[in]  record      削除する予約購入記録
*/
void DeletePrepurchaseRecord(const nn::es::PrepurchaseRecord& record) NN_NOEXCEPT;

/**
*   @brief      全ての Common チケットの削除を行います。
*/
void DeleteAllCommonTicket() NN_NOEXCEPT;

/**
*   @brief      全ての Personalized チケットの削除を行います。
*/
void DeleteAllPersonalizedTicket() NN_NOEXCEPT;

/**
*   @brief      全ての予約購入記録の削除を行います。
*/
void DeleteAllPrepurchaseRecord() NN_NOEXCEPT;

/**
*   @brief      ticketIdExclusionList[] にない全ての Personalized チケットの削除を行います。
*
*   @param[in]  ticketIdExclusionList[] 削除を行わないチケットの TicketId の配列
*   @param[in]  count                   削除を行わないチケットの要素数
*/
void DeleteAllPersonalizedTicket(const nn::es::TicketId ticketIdExclusionList[], int count) NN_NOEXCEPT;

/**
*   @brief      Common チケットの数を取得します。
*
*   @return     取得した Common チケットの数
*/
int CountCommonTicket() NN_NOEXCEPT;

/**
*   @brief      Personalized チケットの数を取得します。
*
*   @return     取得した Personalized チケットの数
*/
int CountPersonalizedTicket() NN_NOEXCEPT;

/**
*   @brief      予約購入記録の数を取得します。
*
*   @return     取得したチケット記録の数
*/
int CountPrepurchaseRecord() NN_NOEXCEPT;

/**
*   @brief      Common チケットの RightsID のリストを取得します。
*
*   @detail     RightsId が同じチケットを複数取得している場合は重複しないように列挙されます。
*
*   @param[out] outRightsIdList[]   取得した Common チケットの 128bit RightsID を格納する配列
*   @param[in]  count               取得した Common チケットの 128bit RightsID を格納する配列の要素数
*
*   @return     取得した Common チケットの RightsID の数
*/
int ListCommonTicketRightsIds(nn::es::RightsIdIncludingKeyId outRightsIdList[], int count) NN_NOEXCEPT;

/**
*   @brief      Personalized チケットの RightsID のリストを取得します。
*
*   @detail     RightsId が同じチケットを複数取得している場合は重複しないように列挙されます。
*
*   @param[out] outRightsIdList[]   取得した Personalized チケットの 128bit RightsID を格納する配列
*   @param[in]  count               取得した Personalized チケットの 128bit RightsID を格納する配列の要素数
*
*   @return     取得した Personalized チケットの RightsID の数
*/
int ListPersonalizedTicketRightsIds(nn::es::RightsIdIncludingKeyId outRightsIdList[], int count) NN_NOEXCEPT;

/**
*   @brief      予約購入記録の RightsID のリストを取得します。
*
*   @detail     RightsId が同じ予約購入記録を複数取得している場合は重複しないように列挙されます。
*
*   @param[out] outRightsIdList[]   取得した予約購入記録の 128bit RightsID を格納する配列
*   @param[in]  count               取得した予約購入記録の 128bit RightsID を格納する配列の要素数
*
*   @return     予約購入記録を取得した RightsID の数
*/
int ListPrepurchaseRecordRightsIds(nn::es::RightsIdIncludingKeyId outRightsIdList[], int count) NN_NOEXCEPT;

/**
*   @brief      TicketIdList[] の中から保有していない Personalized チケットのリストを取得します。
*
*   @param[out] outTicketIdList[]       保有していない Personalized チケットの TicketId を格納する配列
*   @param[in]  outTicketIdListCount    保有していない Personalized チケットの TicketId を格納する配列の要素数
*   @param[in]  TicketIdList[]          Personalized チケットの TicketId を格納する配列
*   @param[in]  ticketIdListCount       Personalized チケットの TicketId を格納する配列の要素数
*
*   @return     取得した Personalized チケットの数
*/
int ListMissingPersonalizedTicket(nn::es::TicketId outTicketIdList[], int outTicketIdListCount, const nn::es::TicketId ticketIdList[], int ticketIdListCount) NN_NOEXCEPT;

/**
*   @brief      Common チケットのサイズを取得します。(最初に見つかったチケットのデータを取得します。)
*
*   @param[out] pOutSize                取得した Common チケットのサイズ
*   @param[in]  rightsId                取得する Common チケットの 128bit RightsId
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess           成功しました。
*   @retval     ResultRightsIdNotFound  RightsId が見つかりませんでした。
*/
Result GetCommonTicketSize(size_t* pOutSize, nn::es::RightsIdIncludingKeyId rightsId) NN_NOEXCEPT;

/**
*   @brief      Common チケットのデータを取得します。(最初に見つかったチケットのデータを取得します。)
*
*   @param[out] pOutSize                取得した Common チケットのバッファサイズ
*   @param[out] pOutBuffer              取得した Common チケットを格納するバッファ
*   @param[in]  size                    取得した Common チケットを格納するバッファのサイズ
*   @param[in]  rightsId                取得する Common チケットの 128bit RightsId
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess           成功しました。
*   @retval     ResultRightsIdNotFound  RightsId が見つかりませんでした。
*   @retval     ResultBufferNotEnough   バッファが不足しています。
*
*   @pre
*       - outBufferSize >= 対象 rightsId に対して es::GetCommonTicketSize() で取得したサイズ or es::MaxTicketSize
*
*/
Result GetCommonTicketData(size_t* pOutSize, void* pOutBuffer, size_t size, nn::es::RightsIdIncludingKeyId rightsId) NN_NOEXCEPT;

/**
*   @brief      暗号化されたチケットのサイズを取得します。(最初に見つかったチケットのデータを取得します。)
*
*   @param[out] pOutSize                取得した暗号化されたチケットのサイズ
*   @param[in]  rightsId                取得するチケットの 128bit RightsId
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess           成功しました。
*   @retval     ResultRightsIdNotFound  RightsId が見つかりませんでした。
*/
Result GetEncryptedTicketSize(size_t* pOutSize, nn::es::RightsIdIncludingKeyId rightsId) NN_NOEXCEPT;

/**
*   @brief      暗号化されたチケットのデータを取得します。(最初に見つかったチケットのデータを取得します。)
*
*   @details    チケットを乱数から生成した 128bit 鍵で AES-CBC (PKCS#7 Padding)暗号化して返却します。
*               暗号化鍵は es ライブラリが保有している 2048bit RSA 公開鍵で OAEP-SHA-256 暗号化して返却します。
*
*   @param[out] pOutTicketId            取得した暗号化されたチケットのチケット ID
*   @param[out] pOutSize                取得した暗号化されたチケットのサイズ
*   @param[out] pOutBuffer              取得した暗号化されたチケットを格納するバッファ
*   @param[in]  outBufferSize           取得した暗号化されたチケットを格納するバッファのサイズ
*   @param[out] pOutKey                 取得したチケットを暗号化した鍵を格納するバッファ
*   @param[in]  outKeySize              取得したチケットを暗号化した鍵を格納するバッファのサイズ
*   @param[in]  rightsId                取得するチケットの 128bit RightsId
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess           成功しました。
*   @retval     ResultRightsIdNotFound  RightsId が見つかりませんでした。
*   @retval     ResultBufferNotEnough   バッファが不足しています。
*
*   @pre
*       - outBufferSize >= 対象 rightsId に対して es::GetEncryptedTicketSize() で取得したサイズ or es::MaxEncryptedTicketSize
*       - outKeySize >= 256
*/
Result GetEncryptedTicketData(TicketId* pOutTicketId, size_t* pOutSize, void* pOutBuffer, size_t outBufferSize, void* pOutKey, size_t outKeySize, nn::es::RightsIdIncludingKeyId rightsId) NN_NOEXCEPT;

/**
*   @brief      チケットがあるかどうかを確認します。
*
*   @param[out] outOwnTicketList[]  チケットを保有していたかの結果を格納する配列
*   @param[in]  rightsIdList[]      チケットがあるか確認する 128bit RightsId の配列
*   @param[in]  count               チケットがあるか確認する 128bit RightsId の配列の要素数
*/
void OwnTicket(bool outOwnTicketList[], const nn::es::RightsIdIncludingKeyId rightsIdList[], int count) NN_NOEXCEPT;

/**
*   @brief      チケットの情報を取得します。
*
*   @param[out] outTicketInfoList[] 取得したチケットの情報を格納する配列
*   @param[in]  ticketInfoListCount 取得したチケットの情報を格納する配列の要素数
*   @param[in]  rightsIdList[]      チケットの情報を取得する 128bit RightsId の配列
*   @param[in]  rightsIdListCount   チケットの情報を取得する 128bit RightsId の配列の要素数
*
*   @return     チケットの情報を取得した数
*/
int GetTicketInfo(nn::es::TicketInfo outTicketInfoList[], int ticketInfoListCount, const nn::es::RightsIdIncludingKeyId rightsIdList[], int rightsIdListCount) NN_NOEXCEPT;

/**
*   @brief      チケットの情報を比較的高速に取得します。
*
*   @param[out] outTicketInfoList[] 取得したチケットの情報を格納する配列
*   @param[in]  ticketInfoListCount 取得したチケットの情報を格納する配列の要素数
*   @param[in]  rightsId            チケットの情報を取得する 128bit RightsId
*
*   @return     チケットの情報を取得した数
*/
int ListLightTicketInfo(nn::es::LightTicketInfo outTicketInfoList[], int ticketInfoListCount, nn::es::RightsIdIncludingKeyId rightsId) NN_NOEXCEPT;

/**
*   @brief      予約購入記録を取得します。
*
*   @param[out] outPrepurchaseRecordList取得した予約購入記録の情報を格納する配列
*   @param[in]  count                   取得した予約購入記録の情報を格納する配列の要素数
*   @param[in]  rightsId                チケット記録の情報を取得する 128bit RightsId
*
*   @return     予約購入記録を取得した数
*/
int ListPrepurchaseRecordInfo(nn::es::PrepurchaseRecord outPrepurchaseRecordList[], int count, nn::es::RightsIdIncludingKeyId rightsId) NN_NOEXCEPT;

/**
*   @brief      Challenge データに対して署名と証明書を取得します。
*
*   @param[out] pOutSign        取得した署名データ
*   @param[out] pOutCertificate 取得した証明書データ
*   @param[in]  pData           Challenge データを格納したポインタ
*   @param[in]  size            Challenge データのサイズ
*/
void SignData(nn::es::Sign* pOutSign, nn::es::Certificate* pOutCertificate, const void* pData, size_t size) NN_NOEXCEPT;

/**
*   @brief      Common チケットと証明書のサイズを取得します。(最初に見つかったチケットのデータを取得します。)
*
*   @param[out] pOutTicketSize          取得した Common チケットのサイズ
*   @param[out] pOutCertificateSize     取得した証明書のサイズ
*   @param[in]  rightsId                取得する Common チケットの 128bit RightsId
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess           成功しました。
*   @retval     ResultRightsIdNotFound  RightsId が見つかりませんでした。
*/
Result GetCommonTicketAndCertificateSize(size_t* pOutTicketSize, size_t* pOutCertificateSize, nn::es::RightsIdIncludingKeyId rightsId) NN_NOEXCEPT;

/**
*   @brief      Common チケットと証明書のデータを取得します。(最初に見つかったチケットのデータを取得します。)
*
*   @param[out] pOutTicketSize          取得した Common チケットのサイズ
*   @param[out] pOutCertificateSize     取得した証明書のサイズ
*   @param[out] pOutTicketBuffer        取得した Common チケットを格納するバッファ
*   @param[out] pOutCertificateBuffer   取得した証明書を格納するバッファ
*   @param[in]  ticketBufferSize        取得した Common チケットを格納するバッファのサイズ
*   @param[in]  certificateBufferSize   取得した証明書を格納するバッファのサイズ
*   @param[in]  rightsId                取得する Common チケットの 128bit RightsId
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess           成功しました。
*   @retval     ResultRightsIdNotFound  RightsId が見つかりませんでした。
*
*   @pre
*       - ticketBufferSize >= 対象 rightsId に対して es::GetCommonTicketAndCertificateSize() で取得したチケットのサイズ or es::MaxTicketSize
*       - certificateBufferSize >= 対象 rightsId に対して es::GetCommonTicketAndCertificateSize() で取得した証明書のサイズ
*/
Result GetCommonTicketAndCertificateData(size_t* pOutTicketSize, size_t* pOutCertificateSize, void* pOutTicketBuffer, void* pOutCertificateBuffer, size_t ticketBufferSize, size_t certificateBufferSize, nn::es::RightsIdIncludingKeyId rightsId) NN_NOEXCEPT;

/**
*   @brief      指定した RightsId のチケットの数を取得します。
*
*   @param[in]  rightsId                取得するチケットの RightsId
*
*   @return     取得したチケットの数
*/
int CountTicket(nn::es::RightsId rightsId) NN_NOEXCEPT;

/**
*   @brief      指定した RightsId のチケットの 128bit RightsId のリストを取得します。
*
*   @detail     RightsId が同じチケットを複数取得している場合は重複しないように列挙されます。
*
*   @param[out] outRightsIdList[]       取得したチケットの 128bit RightsId を格納する配列
*   @param[in]  count                   取得したチケットの 128bit RightsId を格納する配列の要素数
*   @param[in]  rightsId                取得するチケットの RightsId
*
*   @return     取得したチケットの RightsId の数
*/
int ListTicketRightsIds(nn::es::RightsIdIncludingKeyId outRightsIdList[], int count, nn::es::RightsId rightsId) NN_NOEXCEPT;

/**
*   @brief      指定した RightsId の予約購入記録の数を取得します。
*
*   @param[in]  rightsId                取得する予約購入記録の RightsId
*
*   @return     取得した予約購入記録の数
*/
int CountPrepurchaseRecord(nn::es::RightsId rightsId) NN_NOEXCEPT;

/**
*   @brief      指定した RightsId の予約購入記録の 128bit RightsId のリストを取得します。
*
*   @detail     RightsId が同じ予約購入記録を複数取得している場合は重複しないように列挙されます。
*
*   @param[out] outRightsIdList[]   取得した予約購入記録の 128bit RightsId を格納する配列
*   @param[in]  count               取得した予約購入記録の 128bit RightsId を格納する配列の要素数
*   @param[in]  rightsId            取得する予約購入記録の RightsId
*
*   @return     取得した予約購入記録の RightsId の数
*/
int ListPrepurchaseRecordRightsIds(nn::es::RightsIdIncludingKeyId outRightsIdList[], int count, nn::es::RightsId rightsId) NN_NOEXCEPT;

}}

