﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
*   @file
*   @brief  使用中権利リストの管理に関する機能
*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_StaticAssert.h>

namespace nn { namespace es { namespace detail {
    struct ActiveRightsContextHandleObject;
}}}

namespace nn { namespace es {

/**
*   @brief  作成可能な権利リストのコンテキスト最大数を表す定数です。
*/
const int ActiveRightsContextCountMax = 2;


/**
*   @brief  権利リスト格納領域の最大セット数を表す定数。
*
*   @details
*   この数は作成可能な権利コンテキストの最大数よりも小さい値とすることで、
*   各権利コンテキストの権利リスト格納用メモリを削減する効果がある。
*
*   アプリの再起動要求時などでは、以下のようにアプリプロセスの終了と起動は
*   シーケンシャルに行われるものの、2 つの権利コンテキストが同時に存在する
*   瞬間がわずかに存在する。
*
*   現在アプリ A の寿命:  <----|============|->
*   後続アプリ B の寿命:                  <----|=================|->
*                                           ↑
*                                           瞬間的に２つのハンドルが存在
*
*     <--  -->: 権利コンテキストの生存区間
*     |======|: 実アプリプロセスの生存区間
*
*
*   am の実装次第では、権利コンテキストへの権利リスト登録を
*   アプリプロセス起動の直前まで遅延することで、各権利コンテキストに保持する
*   権利リスト用のメモリ領域を削減できる可能性がある。
*
*   TORIAEZU:
*   SIGLO-85070 で UsingRightsArrayCountMax を 1 に最適化するまでは、
*   一旦 ActiveRightsContextCountMax と同じ値にしておく。
*
*/
const int UsingRightsArrayCountMax = ActiveRightsContextCountMax;
NN_STATIC_ASSERT(UsingRightsArrayCountMax <= ActiveRightsContextCountMax);


/**
*   @brief  使用中権利リストのコンテキストを扱うためのハンドル型です。
*/
struct ActiveRightsContextHandle
{
    detail::ActiveRightsContextHandleObject* _p;

    /**
    *   @brief  無効なハンドルを返します。
    */
    static ActiveRightsContextHandle GetInvalidHandle() NN_NOEXCEPT
    {
        ActiveRightsContextHandle handle = { NULL };
        return handle;
    }

    /**
    *   @brief  ハンドルが有効か否かを判定します。
    */
    bool IsValid() const NN_NOEXCEPT
    {
        return !(_p == NULL);
    }
};

/**
*   @brief  使用中権利リストの状態を定義した列挙型です。
*
*   @details
*     使用中権利リストの中で、以下の複数の条件に合致するものがある場合、
*     以下の記載順でより上位のものが優先的に返ります。
*
*/
enum ActiveRightsContextStatus
{
    ActiveRightsContextStatus_NotChecked,   //!< 使用中権利リストの有効・無効判定が行われていません。
    ActiveRightsContextStatus_Active,       //!< 使用中権利リストが利用可能状態です
    ActiveRightsContextStatus_Expired,      //!< 使用中権利リストの中に有効期限が失効したものが含まれています
    ActiveRightsContextStatus_Inactive,     //!< 使用中権利リストの中に利用不可状態のものが含まれています
};


}}  // namespace nn::es
