﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief 内部システム用APIの定義
 */

#pragma once

#include <nn/nn_Result.h>
#include <nn/err/err_SystemTypes.h>
#include <nn/err/err_Types.h>

 /**
 * @file
 * @brief システム向けのエラーAPIの定義
 */

namespace nn { namespace err {

//! @name システム向けのエラーAPI
//! @{

/**
* @brief        nn::Result を nn::err::ErrorCode に変換します。
* @param[in]    result リザルト
* @return       エラーコード
* @pre
*               - result.IsFailure() == true
* @details      エラー表示を行う nn::err::ShowError 関数は nn::Result を受け取るものと nn::err::ErrorCode を受け取るものの
*               両方が用意されており、基本的に nn::Result から nn::err::ErrorCode に変換する必要はありません。
*               本関数は nn::Result で定義されたエラーを独自の方法で表示する必要があり、
*               そこでエラーコードを併せて表示する場合などにご利用ください。
*/
ErrorCode ConvertResultToErrorCode(const Result& result) NN_NOEXCEPT;

/**
* @brief        エラーコードの文字列表現を取得します。
* @param[out]   outBuffer 文字列の格納先
* @param[in]    outBufferLength outBuffer の長さ
* @param[in]    errorCode エラーコード
* @pre
*               - outBuffer != nullptr
*               - outBufferLength >= ErrorCode::StringLengthMax
*/
void GetErrorCodeString(char* outBuffer, size_t outBufferLength, ErrorCode errorCode) NN_NOEXCEPT;

/**
 * @brief        エラーメッセージデータベース（システムデータ）のバージョン情報を取得します。
 * @param[out]   outVersion エラーメッセージデータベースのバージョン
 * @return       処理の結果が返ります。
 * @pre
 *               - outVersion != nullptr
 */
nn::Result GetErrorMessageDatabaseVersion(ErrorMessageDatabaseVersion* outVersion) NN_NOEXCEPT;

/**
* @brief        （未実装）エラービューアが終了パラメータとして設定したジャンプ先へのジャンプを実行します。
* @param[in]    destination ジャンプ先
* @pre
*               - destination != ErrorViewerJumpDestination::Nowhere
* @details      ジャンプを無効にした状態でエラービューアを起動した場合、ジャンプ先の情報がエラービューア呼び出し関数の戻り値として返されます。
*               ジャンプの準備が整い次第、この関数を呼んでジャンプを実行して下さい。
*/
void ExecuteJump(ErrorViewerJumpDestination destination) NN_NOEXCEPT;

//! @}
//-----------------------------------------------------------------------------

}}
