﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <curl/curl.h>
#include <nn/nn_SdkLog.h>

#include <nn/err/err_Types.h>
#include <nn/err/err_Result.h>
#include <nn/ssl/ssl_Types.h>
#include <nn/ssl/ssl_Context.h>
#include <nn/util/util_Uuid.h>
#include <nn/util/util_UuidApi.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/ns/ns_ApplicationManagerApi.h>

namespace nn { namespace err {

/**
* @brief    ダウンロードコンテキストを表す構造体です。
*/
struct DownloadContext
{
    CURL*       pCurl;    //!< Curl オブジェクトです。
    curl_slist* pHeader;  //!< HTTP ヘッダリストです。
    void*       pBuffer;  //!< エラー情報を格納するバッファです。
    size_t      readSize; //!< 読み込んだサイズです。
};

/**
* @brief    サービスステータス情報をダウンロードするクラスです。
*/
class ServiceStatusDownloader
{
public:
    /**
    * @brief    ServiceStatusDownloader クラスのインスタンスを作成します。
    */
    ServiceStatusDownloader() NN_NOEXCEPT{}

    /**
    * @brief    ServiceStatusDownloader クラスのインスタンスを破棄します。
    */
    ~ServiceStatusDownloader() NN_NOEXCEPT{}

    /**
     * @brief        ServiceStatusDownloader クラス初期化します。
     * @pre          nn::socket::Initialize() / curl_global_init() を呼び出し済み。
     * @details      内部で SSL コンテキストの作成を行います。
     */
    void Initialize() NN_NOEXCEPT;

    /**
     * @brief        ServiceStatusDownloader クラスを終了します。
     * @details      内部で作成した SSL コンテキストを破棄します。
     */
    void Finalize() NN_NOEXCEPT;

    /**
     * @brief         障害情報をダウンロードします。
     * @param[out]    pBuffer  service_status.jsonファイルの最大サイズである900KByteのバッファ領域。
     * @param[in]     size     バッファサイズ。(ファイルの最大サイズである900KByteを指定してください)
     * @retresult
     *    @handleresult{nn::ResultSuccess,         成功しました。}
     *    @handleresult{nn::err::ResultHttpError,  ネットワーク、サーバーの障害により、HTTPエラーが発生しました。}
     * @endretresult
     * @pre           nn::nifm::SubmitNetworkRequestAndWait() を実行してネットワークとの接続を確立済み。
     * @details       本関数は内部で curl_easy_perform() を呼び出します。@n
     *                そのため、処理がブロックされてしまうことに注意してください。
     */
    nn::Result Download(char* pBuffer, size_t size) NN_NOEXCEPT;

    /**
     * @brief         ダウンロードをキャンセルします。
     * @details       本関数は、@ref Download を使用して障害情報をダウンロードする時のみ使用可能です。@n
     *                上記の関数を使用するスレッドはブロックされるため、別スレッドから呼び出してください。
     */
    void Cancel() NN_NOEXCEPT;

    /**
     * @brief         ダウンロードコンテキストを作成します。
     * @param[out]    pContext    ダウンロードコンテキスト。
     * @param[out]    pBuffer     service_status.jsonファイルの最大サイズである900KByteのバッファ領域。
     * @param[in]     size        バッファサイズ。(ファイルの最大サイズである900KByteを指定してください)
     * @retresult
     *    @handleresult{nn::ResultSuccess,         成功しました。}
     *    @handleresult{nn::err::ResultHttpError,  ネットワーク、サーバーの障害により、HTTPエラーが発生しました。}
     * @endretresult
     * @pre           curl_global_init() を呼び出し済み。
     * @details       本関数はサービスステータスのダウンロードに必要なコンテキストを作成します。@n
     *                ダウンロード処理は DownloadContext::pCurl を使用して curl_easy_perform() / curl_multi_perform() を実行してください。@n
     *                本関数内ではDownloadContext.pHeaderに"Connection: Keep-Alive"を設定しています。
     */
    nn::Result CreateDownloadContext(DownloadContext* pContext, void* pBuffer, size_t size) NN_NOEXCEPT;

    /**
     * @brief         ダウンロードコンテキストを破棄します。
     * @param[in]     pContext        ダウンロードコンテキスト。
     */
    void DestroyDownloadContext(DownloadContext* pContext) NN_NOEXCEPT;


    /**
     * @brief         ダウンロード処理の結果をチェックします。
     * @retresult
     *    @handleresult{nn::ResultSuccess,         成功しました。}
     *    @handleresult{nn::err::ResultHttpError,  ネットワーク、サーバーの障害により、HTTPエラーが発生しました。}
     * @endretresult
     * @details       本関数はダウンロード処理後に呼び出してください。
     */
    nn::Result CheckDownloadResult() const NN_NOEXCEPT;


private:
    /**
     * @brief         エラーリザルトを設定します。
     * @param[in]     result    設定する Result 値。
     */
    void SetLastResult(nn::Result result) NN_NOEXCEPT;

    /**
     * @brief         直近のエラーリザルトを取得します。
     * @return        直近のエラーリザルト。
     */
    nn::Result GetLastResult() const NN_NOEXCEPT;

    /**
     * @brief         エラーリザルトをリセットします。
     */
    void ResetLastResult() NN_NOEXCEPT;

private:
    nn::Result          m_LastResult;
    DownloadContext*    m_pDownloadContext;
    nn::ssl::Context    m_SslContext;
    CURLcode            m_ErrorCode;
    bool                m_IsCanceled;
};

}}
