﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_ApplicationId.h>
#include <nn/nn_Result.h>
#include <nn/err/err_ErrorContext.h>
#include <nn/err/err_SystemTypes.h>
#include <nn/err/err_Types.h>

namespace nn { namespace err {

//! @name SDK ツール・アプリケーション向けの API
//! @{

/**
* @brief        エラーレポートを作成します。
* @param[in]    applicationId エラーレポートの作成元（＝呼び出し元自身）アプリケーションID
* @param[in]    result エラーの内容を表す Result。
*/
void ReportError(nn::ApplicationId applicationId, nn::Result result) NN_NOEXCEPT;

/**
* @brief        エラーレポートを作成します。
* @param[in]    applicationId エラーレポートの作成元（＝呼び出し元自身）のアプリケーションID
* @param[in]    errorCode エラーの内容を表す ErrorCode。
*/
void ReportError(nn::ApplicationId applicationId, ErrorCode errorCode) NN_NOEXCEPT;

/**
* @brief        エラーレポートを作成します。
* @param[in]    applicationId エラーレポートの作成元（＝呼び出し元自身）のアプリケーションID
* @param[in]    result エラーの内容を表す Result。
* @param[in]    resultBacktrace エラーレポートに保存する Result のバックトレース。
*/
void ReportError(nn::ApplicationId applicationId, nn::Result result, const ResultBacktrace& resultBacktrace) NN_NOEXCEPT;

/**
* @brief        エラーレポートを作成します。
* @param[in]    applicationId エラーレポートの作成元（＝呼び出し元自身）のアプリケーションID
* @param[in]    errorCode エラーの内容を表す Result。
* @param[in]    resultBacktrace エラーレポートに保存する Result のバックトレース。
*/
void ReportError(nn::ApplicationId applicationId, ErrorCode errorCode, const ResultBacktrace& resultBacktrace) NN_NOEXCEPT;

/**
* @brief        エラーレポートを作成します。
* @param[in]    applicationId エラーレポートの作成元（＝呼び出し元自身）アプリケーションID
* @param[in]    result エラーの内容を表す Result。
* @param[in]    errorContext エラーレポートに保存するエラーコンテキスト。
* @param[in]    workBuffer ワークバッファ。
* @param[in]    workBufferSize workBuffer のサイズ。
* @pre
*               - workBuffer != nullptr
*               - workBufferSize >= 1024
*/
void ReportError(nn::ApplicationId applicationId, nn::Result result, const ErrorContext& errorContext, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT;

/**
* @brief        エラーレポートを作成します。
* @param[in]    applicationId エラーレポートの作成元（＝呼び出し元自身）アプリケーションID
* @param[in]    errorCode エラーの内容を表す ErrorCode。
* @param[in]    errorContext エラーレポートに保存するエラーコンテキスト。
* @param[in]    workBuffer ワークバッファ。
* @param[in]    workBufferSize workBuffer のサイズ。
* @pre
*               - workBuffer != nullptr
*               - workBufferSize >= 1024
*/
void ReportError(nn::ApplicationId applicationId, ErrorCode errorCode, const ErrorContext& errorContext, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT;

//! @}

}}
