﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief ライブラリ/モジュール向けエラー管理用APIの定義
 */

#pragma once

#include <nn/err/err_Types.h>

namespace nn { namespace err {

//! @name ライブラリ/モジュール向けエラー管理用API
//! @{

/**
 * @brief ライブラリや本体機能の独自エラーを ErrorCode に変換します。
 * @param[in] errorCodeCategory エラーのカテゴリ。ライブラリや本体機能の識別子。
 * @param[in] errorCodeNumber エラーの番号。
 * @pre
 *  - errorCodeCategory != 0
 * @details
 *  - 本関数は nn::Result 値を用いないライブラリのエラーをエラービューアーで表示させたい場合に使用します。
 *    以下のように、 ErrorCode の生成とエラー表示を一括で行うことで、
 *    ライブラリ利用者に対しては ErrorCode を見せないように実装することができます。
 *    @code
 *    nn::Result ShowMyLibError(MyLibError errValue)
 *    {
 *        // 独自エラーに対応する category と number を取得(生成)
 *        nn::err::ErrorCodeCategory category = GetMyLibErrorCategory(errValue);
 *        nn::err::ErrorCodeNumber number = GetMyLibErrorNumber(errValue);
 *        // エラーコードを生成
 *        ErrorCode errCode = nn::err::MakeErrorCode(category, number);
 *        // エラーを表示
 *        return nn::err::ShowError(errcode);
 *    }
 *    @endcode
 *  - ライブラリ側の実装により、 ErrorCode を直接アプリケーションに返す実装を採ることも可能ですが、
 *    その場合はアプリケーション側で ShowError(ErrorCode) を呼び出す実装を行うように案内してください。
 *    (ErrorCode はエラービューアー表示以外の用途で使われることを想定していませんのでご注意ください。)
 */
ErrorCode MakeErrorCode(ErrorCodeCategory errorCodeCategory, ErrorCodeNumber errorCodeNumber) NN_NOEXCEPT;

//! @}
//-----------------------------------------------------------------------------

}} // namespace nn::err
