﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

 /**
 * @file
 * @brief エラービューアアプレットの起動・終了パラメータの定義。
 */

#pragma once

#include <nn/err/err_Types.h>
#include <nn/err/err_SystemTypes.h>
#include <nn/nn_Common.h>
#include <nn/nn_StaticAssert.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/os.h>
#include <nn/settings/system/settings_Region.h>
#include <nn/time/time_PosixTime.h>

namespace nn { namespace err {

/**
* @brief    メッセージ指定でエラーを表示する際に指定できるメッセージの最大バッファサイズ。
*/
const int ErrorViewerStartupParamMessageBufferSizeMax = 2048;

/**
* @brief    1つの EULA データの最大サイズです。
*/
const size_t EulaDataSizeMax = 16 * 1024 - 24;

/**
* @brief    EULA 表示呼び出し時に与えられる最大データ数です。
*/
const int EulaDataCountMax = 8;

/**
* @brief    エラービューアの起動パラメータのアラインメント。
*/
const size_t ErrorViewerStartupParamAddressAlignment = std::alignment_of<std::max_align_t>::value;

/**
* @brief    全てのエラービューアアプレット呼び出し API で共通で設定するエラービューアアプレット起動パラメータを表します。
*           常に起動パラメータの先頭に配置します。
*/
struct ErrorViewerStartupParamCommon
{
    ErrorType                           errorType;                      //!< エラーの種類。
    bool                                isJumpEnabled;                  //!< エラービューアアプレットから他のアプレットへのジャンプが有効かどうかのフラグ。
    Bit8                                reserved[2];

    ErrorViewerStartupParamCommon(nn::err::ErrorType _errorType, bool _isJumpEnabled = true)
        : errorType(_errorType), isJumpEnabled(_isJumpEnabled), reserved{} {};
};
NN_STATIC_ASSERT(sizeof(ErrorViewerStartupParamCommon) == 4);

/**
* @brief    ErrorType::SystemData の際に渡されるパラメータを表します。
*/
struct ErrorViewerStartupParamForSystemData
{
    ErrorViewerStartupParamCommon       common;
    uint8_t                             version;                        //!< パラメータ構造体のバージョン（注意：初期化していなかったため、この値を処理の分岐に用いることはできない）
    bool                                hasResultBacktrace;             //!< ResultBackTrace の有無。true の場合、2番目のストレージに ResultBackTrace が入力されています（現時点では ErrorContext との両用は不可）。
    bool                                isErrorCode;                    //!< エラーコードとリザルトのどちらが有効かどうかを示すフラグです。
    bool                                hasErrorContext;                //!< ErorContext の有無。true の場合、2番目のストレージに ErrorContext が入力されています（現時点では ResultBacktrace との両用は不可）。
    ErrorCode                           errorCode;                      //!< エラーコード。isErrorCode == true のとき、システムデータの読み取りに使用されます。
    nn::Result                          result;                         //!< リザルト。isErrorCode == false のとき、エラーコードに変換の上でシステムデータの読み取りに使用されます。
    ErrorViewerStartupParamForSystemData()
        : common(nn::err::ErrorType::SystemData), hasResultBacktrace(false), isErrorCode(false), hasErrorContext(false) {};

    void SetErrorCode(const ErrorCode& errorCode) NN_NOEXCEPT;
    void SetResult(const nn::Result& result) NN_NOEXCEPT;

    /**
    * @brief    エラーコードを取得します。エラービューアアプレット向けの関数です。
    * @details  格納している情報が Result だった場合（isErrorCode == false）、Result を変換した ErrorCode を返します。
    */
    ErrorCode GetErrorCode() const NN_NOEXCEPT;
};
NN_STATIC_ASSERT(sizeof(ErrorViewerStartupParamForSystemData) == 20);
NN_STATIC_ASSERT(sizeof(ErrorViewerStartupParamForSystemData) <= ErrorViewerStartupParamSizeMax);

/**
* @brief    ErrorType::SystemError の際に渡されるパラメータを表します。
*/
struct ErrorViewerStartupParamForSystemError
{
    ErrorViewerStartupParamCommon       common;
    uint8_t                             version;                        //!< パラメータ構造体のバージョン。
    bool                                hasErrorContext;                //!< ErorContext の有無。true の場合、2番目のストレージに ErrorContext が入力されています。
    Bit8                                reserved[2];
    ErrorCode                           errorCode;                      //!< エラーコード。表示に使用されます。
    nn::settings::LanguageCode          languageCode;                   //!< エラービューアアプレットのボタン等の UI 部品に使用する言語。
    char                                dialogMessage[ErrorViewerStartupParamMessageBufferSizeMax];       //!< ダイアログ表示用のメッセージ（UTF-8）。
    char                                fullScreenMessage[ErrorViewerStartupParamMessageBufferSizeMax];   //!< 全画面表示用のメッセージ（UTF-8）。

    ErrorViewerStartupParamForSystemError()
        : common(nn::err::ErrorType::SystemError), version(0), hasErrorContext(false), reserved{} {};
};
NN_STATIC_ASSERT(sizeof(ErrorViewerStartupParamForSystemError) == 4120); // 24 + 2048 * 2
NN_STATIC_ASSERT(sizeof(ErrorViewerStartupParamForSystemError) <= ErrorViewerStartupParamSizeMax);

/**
* @brief    ErrorType::ApplicationError の際に渡されるパラメータを表します。
*/
struct ErrorViewerStartupParamForApplicationError
{
    ErrorViewerStartupParamCommon       common;
    uint8_t                             version;                        //!< パラメータ構造体のバージョン。
    Bit8                                reserved[3];
    ApplicationErrorCodeNumber          applicationErrorCodeNumber;     //!< アプリケーションエラーコードの番号。
    nn::settings::LanguageCode          languageCode;                   //!< エラービューアアプレットのボタン等の UI 部品に使用する言語。
    char                                dialogMessage[ErrorViewerStartupParamMessageBufferSizeMax];       //!< ダイアログ表示用のメッセージ（UTF-8）。
    char                                fullScreenMessage[ErrorViewerStartupParamMessageBufferSizeMax];   //!< 全画面表示用のメッセージ（UTF-8）。

    ErrorViewerStartupParamForApplicationError()
        : common(nn::err::ErrorType::ApplicationError), version(0), reserved{} {};
};
NN_STATIC_ASSERT(sizeof(ErrorViewerStartupParamForApplicationError) == 4116); // 24 + 2048 * 2
NN_STATIC_ASSERT(sizeof(ErrorViewerStartupParamForApplicationError) <= ErrorViewerStartupParamSizeMax);

/**
* @brief    ErrorType::Eula の際に渡されるパラメータを表します。
*/
struct ErrorViewerStartupParamForEula
{
    ErrorViewerStartupParamCommon       common;
    uint8_t                             version;                        //!< パラメータ構造体のバージョン。
    Bit8                                reserved[3];
    settings::system::RegionCode        regionCode;                     //!< リージョンコード。

    ErrorViewerStartupParamForEula()
        : common(nn::err::ErrorType::Eula), version(0), reserved{} {};
};
NN_STATIC_ASSERT(sizeof(ErrorViewerStartupParamForEula) == 12);
NN_STATIC_ASSERT(sizeof(ErrorViewerStartupParamForEula) <= ErrorViewerStartupParamSizeMax);

/**
* @brief    ErrorType::SystemUpdateEula の際に渡されるパラメータを表します。
*/
struct ErrorViewerStartupParamForSystemUpdateEula
{
    ErrorViewerStartupParamCommon   common;
    uint8_t                         version;                        //!< パラメータ構造体のバージョン。
    Bit8                            reserved[3];
    settings::system::RegionCode    regionCode;                     //!< リージョンコード。
    ErrorViewerStartupParamForSystemUpdateEula()
        : common(nn::err::ErrorType::SystemUpdateEula), version(0), reserved{} {};
};
NN_STATIC_ASSERT(sizeof(ErrorViewerStartupParamForSystemUpdateEula) == 12);
NN_STATIC_ASSERT(sizeof(ErrorViewerStartupParamForSystemUpdateEula) <= ErrorViewerStartupParamSizeMax);

/**
* @brief    ErrorType::SystemUpdateEula の際に呼び出し元から追加で渡される EULA のデータを表します。
*/
struct EulaData
{
    int dataCount;  //!< 有効な EULA データの数。
    Bit8 reserved[60];
    struct
    {
        settings::Language  language;               //<! EULA データの言語。
        Bit8                reserved[4];
        uint64_t            size;                   //<! EULA データのサイズ。
        Bit8                body[EulaDataSizeMax];  //<! EULA データ。
    } data[EulaDataCountMax];

    EulaData() : dataCount(0), reserved{}, data{} {};
};
NN_STATIC_ASSERT(sizeof(EulaData) % nn::os::MemoryPageSize == 0); // LargeStorage の制約
NN_STATIC_ASSERT(sizeof(EulaData) == 128 * 1024);

/**
* @brief    ErrorType::ParentalControl の際に渡されるパラメータを表します。
*/
struct ErrorViewerStartupParamForParentalControl
{
    ErrorViewerStartupParamCommon       common;
    uint8_t                             version;                        //!< パラメータ構造体のバージョン。
    Bit8                                reserved[3];
    nn::Result                          pctlResultRestricted;           //!< ペアレンタルコントロールで制限された機能に対応した Result。

    ErrorViewerStartupParamForParentalControl()
        : common(nn::err::ErrorType::ParentalControl), version(0), reserved{} {};
};
NN_STATIC_ASSERT(sizeof(ErrorViewerStartupParamForParentalControl) == 12);
NN_STATIC_ASSERT(sizeof(ErrorViewerStartupParamForParentalControl) <= ErrorViewerStartupParamSizeMax);

/**
* @brief    ErrorType::RecordedSystemData の際に渡されるパラメータを表します。
*/
struct ErrorViewerStartupParamForRecordedSystemData
{
    ErrorViewerStartupParamCommon       common;
    uint8_t                             version;                        //!< パラメータ構造体のバージョン。
    Bit8                                reserved[3];
    ErrorCode                           errorCode;                      //!< エラーコード。システムデータの読み取りに使用されます。
    nn::time::PosixTime                 timeOfOccurrence;               //!< エラーの発生日時。

    ErrorViewerStartupParamForRecordedSystemData()
        : common(nn::err::ErrorType::RecordedSystemData), version(0), reserved{} {};
};
NN_STATIC_ASSERT(sizeof(ErrorViewerStartupParamForRecordedSystemData) == 24);
NN_STATIC_ASSERT(sizeof(ErrorViewerStartupParamForRecordedSystemData) <= ErrorViewerStartupParamSizeMax);

/**
* @brief    ErrorType::RecordedSystemError の際に渡されるパラメータを表します。
*/
struct ErrorViewerStartupParamForRecordedSystemError
{
    ErrorViewerStartupParamCommon       common;
    uint8_t                             version;                        //!< パラメータ構造体のバージョン。
    Bit8                                reserved[3];
    ErrorCode                           errorCode;                      //!< エラーコード。表示に使用されます。
    nn::time::PosixTime                 timeOfOccurrence;               //!< エラーの発生日時。
    char                                message[ErrorViewerStartupParamMessageBufferSizeMax];   //!< 表示用のメッセージ（UTF-8）。

    ErrorViewerStartupParamForRecordedSystemError()
        : common(nn::err::ErrorType::RecordedSystemError), version(0), reserved{} {};
};
NN_STATIC_ASSERT(sizeof(ErrorViewerStartupParamForRecordedSystemError) == 2072); // 24 + 2048
NN_STATIC_ASSERT(sizeof(ErrorViewerStartupParamForRecordedSystemError) <= ErrorViewerStartupParamSizeMax);

/**
* @brief    ErrorType::RecordedApplicationError の際に渡されるパラメータを表します。
*/
struct ErrorViewerStartupParamForRecordedApplicationError
{
    ErrorViewerStartupParamCommon       common;
    uint8_t                             version;                        //!< パラメータ構造体のバージョン。
    Bit8                                reserved0[3];
    ApplicationErrorCodeNumber          applicationErrorCodeNumber;     //!< アプリケーションエラーコードの番号。
    Bit8                                reserved1[4];
    nn::time::PosixTime                 timeOfOccurrence;               //!< エラーの発生日時。
    ns::ApplicationErrorCodeCategory    applicationErrorCodeCategory;   //!< アプリケーションエラーコードのカテゴリー。
    char                                message[ErrorViewerStartupParamMessageBufferSizeMax];   //!< 表示用のメッセージ（UTF-8）。

    ErrorViewerStartupParamForRecordedApplicationError()
        : common(nn::err::ErrorType::RecordedApplicationError), version(0), reserved0{}, reserved1{} {};
};
NN_STATIC_ASSERT(sizeof(ErrorViewerStartupParamForRecordedApplicationError) == 2080); // 32 + 2048
NN_STATIC_ASSERT(sizeof(ErrorViewerStartupParamForRecordedApplicationError) <= ErrorViewerStartupParamSizeMax);

/**
* @brief        エラーレポートに記録された情報からエラー履歴表示用のパラメータを構築します。
* @param[out]   outBuffer 構築された起動パラメータ。
* @param[out]   outActualSize 構築された起動パラメータのサイズ。
* @param[in]    outBufferSize outBuffer のサイズ。
* @param[in]    errorCodeStr エラーコード文字列。erpt::FieldId::ErrorCode に記録された文字列。
* @param[in]    message エラーメッセージ。erpt::FieldId::ErrorDescription に記録された文字列。ErrorDescription が保存されていない場合は nullptr。
* @param[in]    timeOfOccurrence エラー発生時刻。erpt::FieldId::OccurrenceTimestamp に記録された値。
* @pre
*               - outBuffer != nullptr
*               - outBuffer が ErrorViewerStartupParamAddressAlignment にアラインしている
*               - outBufferSize が ErrorViewerStartupParamSizeMax 以上
* @post
*               - 返り値が true の場合、err::ShowErrorRecord 関数に outBuffer と outActualSize を渡すとエラーの表示ができる。
* @return       パラメータの構築に成功した場合は true。不正な入力などにより失敗した場合は false。
*/
bool CreateErrorViewerStartupParamForRecordedError(void* outBuffer, size_t* outActualSize, size_t outBufferSize, const char* errorCodeStr, const char* message, nn::time::PosixTime timeOfOccurrence) NN_NOEXCEPT;

/**
* @brief    エラービューアアプレットの終了パラメータを表します。
*/
struct ErrorViewerReturnValue
{
    uint8_t                         version;        //!< パラメータ構造体のバージョン。
    ErrorViewerJumpDestination      destination;    //!< ジャンプ先
};

//! @}
//-----------------------------------------------------------------------------

}}
