﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief システムアプレット/ライブラリアプレット向けAPI
 */

#pragma once

#include <nn/nn_Result.h>
#include <nn/err/err_SystemTypes.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace err {


//! @name アプレット向けの API
//! @{

/**
* @brief        エラーを障害解析用に保存します。
* @param[in]    result エラーの内容を示すリザルト。
* @param[in]    applicationId エラーを発生させたアプリケーションのID。
* @param[in]    resultBacktrace エラーの詳細な内容を示すリザルトのバックトレース。
* @param[in]    workBuffer エラーレポートの作成に使用するワークバッファ。resultBacktrace を指定しない場合は nullptr 可。
* @param[in]    workBufferSize workBuffer のサイズ。
* @pre
*               - resultBacktrace == true のとき
*                   - workBuffer が sizeof(uint32_t) にアラインしている
*                   - workBufferSize >= sizeof(uint32_t) * resultBacktrace->count
*               - ns::InitializeForApplicationManager() を呼び出し済み
* @details      保存したエラーはエラー履歴での閲覧対象になりません。
*/
void RecordErrorForTelemetry(Result result, ncm::ApplicationId applicationId, const util::optional<ResultBacktrace>& resultBacktrace, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT;

//! @}
//-----------------------------------------------------------------------------



//! @name ライブラリアプレット向けの API
//! @{

/**
* @brief        エラーを保存します。
* @param[in]    errorCode エラーコード。
* @param[in]    message メッセージ。nullptr もしくは "" を指定した場合は保存されません。
* @param[in]    workBuffer エラーレポートの作成に使用するワークバッファ
* @param[in]    workBufferSize workBuffer のサイズ
* @pre
*               - ライブラリアプレットからの呼び出し。
*               - workBufferSize >= RequiredBufferSizeForRecordError
*               - ns::InitializeForApplicationManager() を呼び出し済み
* @details      エラービューアを使用せずにエラー表示を行うライブラリアプレット向けの機能です。
*               保存したエラーはエラー履歴での閲覧対象になります。
*/
void RecordError(ErrorCode errorCode, const char* message, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT;

/**
* @brief        エラーを障害解析用に保存します。
* @param[in]    errorCode エラーコード。
* @param[in]    message メッセージ。nullptr もしくは "" を指定した場合は保存されません。
* @param[in]    workBuffer エラーレポートの作成に使用するワークバッファ
* @param[in]    workBufferSize workBuffer のサイズ
* @pre
*               - ライブラリアプレットからの呼び出し。
*               - workBufferSize >= RequiredBufferSizeForRecordError
*               - ns::InitializeForApplicationManager() を呼び出し済み
* @details      エラービューアを使用せずにエラー表示を行うライブラリアプレット向けの機能です。
*               保存したエラーはエラー履歴での閲覧対象になりません。
*/
void RecordErrorForTelemetry(ErrorCode errorCode, const char* message, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT;

//! @}
//-----------------------------------------------------------------------------

}}
