﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/err/err_ErrorInfo.h>
#include <nn/err/err_Types.h>

namespace nn { namespace err { namespace detail {

#define NN_ERR_DETAIL_SYSTEM_DATA_MOUNT_NAME "err"

const ErrorCodeNumber DefaultMessageErrorNumberId = 9999;           //!< カテゴリ汎用のメッセージのデスクリプション番号。
const ErrorCode CategoryDefaultNotFoundErrorCode = { 2999, 9999 };  //!< カテゴリの汎用メッセージが存在しない場合に参照するエラーコード（カテゴリ自体が存在しない場合も含む）

/**
* @brief エラーメッセージのシステムデータがエラー情報に設定可能なフラグです。
*/
struct AttributeFlag
{
    //!< 動作中のアプリケーションのバージョンが許容できないことを示します。
    typedef util::BitFlagSet<32, AttributeFlag>::Flag<0> UnacceptableApplicationVersion;
};

typedef struct util::BitFlagSet<32, AttributeFlag> AttributeFlagSet;

// システムデータの各エラーが共通で持つファイルのデータ構造
struct CommonData
{
    uint8_t dialogButtonCount;
    uint8_t dialogButtonAction[3];
    uint8_t fullScreenButtonCount;
    uint8_t fullScreenButtonAction[3];
    uint8_t server;
    char serverCode[9];
    NN_PADDING2;
    ErrorCode dialogMessageReference;
    ErrorCode dialogButtonMessageReference;
    ErrorCode fullScreenMessageReference;
    ErrorCode fullScreenButtonMessageReference;
    uint32_t  attributeFlagSetValue;
};
NN_STATIC_ASSERT(sizeof(CommonData) == 56);

enum class MessageKind : Bit8
{
    Dialog,
    DialogButton,
    FullScreen,
    FullScreenButton,
};

void MakeErrorInfoModuleDirectoryPath(char* outBuffer, size_t bufferLength, ErrorCodeCategory module) NN_NOEXCEPT;
void MakeErrorInfoDirectoryPath(char* outBuffer, size_t bufferLength, ErrorCode errorCode) NN_NOEXCEPT;
void MakeErrorInfoCommonFilePath(char* outBuffer, size_t bufferLength, ErrorCode errorCode) NN_NOEXCEPT;
void MakeErrorInfoMessageFilePath(char* outBuffer, size_t bufferLength, ErrorCode errorCode, nn::settings::LanguageCode langCode, MessageKind messageKind) NN_NOEXCEPT;

bool ErrorMessageDataExists(ErrorCode errorCode) NN_NOEXCEPT;
bool CategoryExists(ErrorCodeCategory category) NN_NOEXCEPT;
bool DefaultErrorMessageDataExists(ErrorCodeCategory module) NN_NOEXCEPT;

nn::Result ReadMessageFile(char16_t* outBuffer, int* outMessageLength, size_t bufferSize, ErrorCode errorCode, nn::settings::LanguageCode langCode, MessageKind messageKind) NN_NOEXCEPT;

void ReadVersion(ErrorMessageDatabaseVersion* outVersion) NN_NOEXCEPT;

void ReadMessageFile(char16_t* outBuffer, size_t bufferSize, const char* key, const nn::settings::LanguageCode& languageCode) NN_NOEXCEPT;

}}}
