﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 *  @file
 *  @brief Error report service types.
*/

#include <nn/os.h>
#include <nn/util/util_BitFlagSet.h>
#include <nn/util/util_Uuid.h>
#include <nn/time/time_PosixTime.h>
#include <nn/erpt/common/erpt_Ids.h>

namespace nn   {
namespace erpt {

#define GENERATE_FIELD(FIELD,ENUM,...)            FIELD=ENUM,
#define GENERATE_CATEGORY(CATEGORY,ENUM,LOGDEPTH) CATEGORY=ENUM,
#define GENERATE_TYPE(TYPE,ENUM)                  TYPE=ENUM,

enum FieldType
{
    FOREACH_TYPE(GENERATE_TYPE)
    FieldType_Last
};

enum FieldId
{
    FOREACH_ID(GENERATE_FIELD)
    FieldId_Last
};

enum CategoryId
{
    FOREACH_CATEGORY(GENERATE_CATEGORY)
    CategoryId_Last
};

#undef GENERATE_FIELD
#undef GENERATE_CATEGORY
#undef GENERATE_TYPE
#undef FOREACH_TYPE
#undef FOREACH_ID
#undef FOREACH_CATEGORY

const uint32_t FieldsPerContext      = 20;              //!< Number of entries per context.
const uint32_t ReportMetaDataLength  = 32;              //!< Max number of bytes in user provided metadata.
const uint32_t NumberOfReports       = 50;              //!< Number of reports to store in the system.
const uint32_t ReportIdLength        = 20;              //!< Size of nn::erpt::ReportId.
const uint32_t ArrayBufferLength     = 256;             //!< Size of the default buffer for string/array type fields
const uint32_t MaxArrayBufferLength  = (96 * 1024);     //!< Max size of user provided buffer for string/array type fields
const uint32_t MaxArrayFieldSize     = (16 * 1024) - 1; //!< Max size of the string/array type fields (size = sizeof(element) * elementCount)

/**
 *  @brief  Supported report types
 */
enum ReportType
{
    ReportType_First     = 0,
    ReportType_Visible   = ReportType_First,        //!< Report may be displayed in the error viewer applet
    ReportType_Invisible = 1,                       //!< Report should not be displayed in the error viewer applet
    ReportType_Last      = 2,
    ReportType_Any       = ReportType_Last
};

/**
 *  @brief  User supplied meta data, may be stored together with the report.
 */
struct ReportMetaData
{
    uint8_t userData[ReportMetaDataLength];
};

/**
 *  @brief  Report identifier, generated by error report service.
 */
struct ReportId
{
    union
    {
        uint8_t     id[ReportIdLength];
        // Launch NUP format.
        util::Uuid  uuidRFC4122;
        #pragma pack(push, 1)
        // Pre-Install format.
        struct
        {
            uint32_t timeLow   : 32;                //!< bits 0-31  of timestamp
            uint16_t timeMid   : 16;                //!< bits 32-48 of timestamp
            uint16_t hiVersion : 16;                //!< random generated bits
            uint8_t  clockHi   : 8;                 //!< bits 48-55 of timestamp
            uint8_t  clockLow  : 8;                 //!< bits 55-63 of timestamp
            uint64_t node      : 48;                //!< random generated bits
        }
        uuid;
        #pragma pack(pop)
    }
    u;
};
NN_STATIC_ASSERT(util::Uuid::Size <= ReportIdLength);

/**
 *  @brief  Report flag bit definitions.
 */
struct ReportFlag
{
    typedef nn::util::BitFlagSet<32, ReportFlag>::Flag<0> Transmitted; //!< Already Transmitted

};

/**
 *  @brief  Report flags.
 */
typedef struct nn::util::BitFlagSet<32, ReportFlag> ReportFlagSet;

/**
 *  @brief  Description of a single report
 */
struct ReportInfo
{
    ReportType          reportType;
    ReportId            reportId;                   //!< Identifier
    ReportMetaData      reportMetaData;             //!< User provided meta data
    ReportFlagSet       reportFlags;                //!< Flags
    nn::time::PosixTime reportTimeStamp;            //!< Timestamp, user clock based
    nn::time::PosixTime reportTimeStampNet;         //!< Timestamp, network clock based
    int64_t             reportSize;
    uint64_t            reserved[3];
};

/**
 *  @brief  List of all stored reports
 */
struct ReportList
{
    uint32_t            reportCount;                //!< Number of valid reports in this list
    ReportInfo          Report[NumberOfReports];    //!< Description for all reports
};

struct FieldEntry
{
    FieldId   id;
    FieldType type;
    union
    {
        uint64_t numericFieldU64;
        uint32_t numericFieldU32;
        uint16_t numericFieldU16;
        uint8_t  numericFieldU8;
        int64_t  numericFieldI64;
        int32_t  numericFieldI32;
        int16_t  numericFieldI16;
        int8_t   numericFieldI8;
        bool     boolField;
        struct
        {
            uint32_t startIndex;
            uint32_t size;
        }
        arrayField;
    }
    u;
};

struct ContextEntry
{
    uint32_t   version;
    uint32_t   fieldCount;
    CategoryId category;
    FieldEntry fields[FieldsPerContext];
    uint8_t*   pArrayBuffer;
    uint32_t   arrayFreeCount;
    uint32_t   arrayBufferSize;
};

struct StorageUsageStatistics
{
    util::Uuid journalUuid;
    uint32_t   usedStorageSize;
    int64_t    maxReportSize;
    uint32_t   reportCount[ReportType_Last];
    uint32_t   transmittedCount[ReportType_Last];
    uint32_t   untransmittedCount[ReportType_Last];
};

}}
