﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief ショップサービスアクセスユーティリティ機能用デバッグ支援機能関連宣言。
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/ec/ec_Result.h>
#include <nn/ec/ec_ShopServiceAccessorTypes.h>

namespace nn { namespace ec {

/**
 * @brief   デバッグ支援機能クラス。
 *
 * @details 本機能は以下の支援機能を提供します。@n
 *
 *          @li デバッグレスポンス支援機能。@n
 *              登録されたキー内容が @ref nn::ec::ShopServiceAccessor 経由で正規要求された要求内容と合致した場合に、
 *              対応するデバッグレスポンスを対象要求の結果として上書き返却します。@n
 *              合致照合のロジックは以下の通りです。@n
 *
 *              @li 登録キー[接続先ショップサービスサーバー種別]が、正規要求の[接続先ショップサービスサーバー種別]と合致した場合。
 *              @li 登録キー[URLパス文字列]が、正規要求の[URLパス文字列]と前方一致した場合。@n
 *                  [URLパス文字列]の照合は、正規要求の[URLパス文字列]がエイリアスシンボルリプレースされた状態に対して行います。@n
 *                  デバッグレスポンス側の[URLパス文字列]も @ref nn::ec::ShopServiceAccessor 同様のエイリアスシンボルリプレースをサポートしています。
 *
 * @note    本機能はデバッグ用です。@n
 *          本デバッグAPIの利用制約として以下の状況での利用は推奨されません。@n
 *
 *          @li @ref nn::ec::ShopServiceAccessor 機能を利用しない実行可能なプログラムからのデバッグAPIの利用の場合。@n
 *              @ref nn::ec::ShopServiceAccessor 機能を利用する実行可能なプログラムが実行中での利用は推奨されません。
 *
 *          @li @ref nn::ec::ShopServiceAccessor 機能を利用する実行可能なプログラムからのデバッグAPIの利用の場合。@n
 *              @ref nn::ec::ShopServiceAccessor 機能が有効な区間( @ref nn::ec::InitializeForShopServiceAccessors() ～ @ref nn::ec::FinalizeForShopServiceAccessors() 間 )中の利用は推奨されません。
 *
 *          利用制約に違反した場合、デバッグAPIもしくは @ref nn::ec::ShopServiceAccessor 機能を利用したタイミングにより、相互でリソース不足のエラーを返す可能性があります。@n
 */
class DebugForShopServiceAccessor
{
    NN_DISALLOW_COPY(DebugForShopServiceAccessor);
    NN_DISALLOW_MOVE(DebugForShopServiceAccessor);

private:
    DebugForShopServiceAccessor() NN_NOEXCEPT = delete;
    ~DebugForShopServiceAccessor() NN_NOEXCEPT = delete;

public:
    /**
     * @brief   スコープサイクル依存デバッグ支援機能有効期間セッションハンドル。
     */
    class ScopedSession
    {
        NN_DISALLOW_COPY(ScopedSession);
        NN_DISALLOW_MOVE(ScopedSession);

    public:
        /**
         * @brief   デフォルトコンストラクタ。
         */
        NN_IMPLICIT ScopedSession() NN_NOEXCEPT;

        /**
         * @brief   デストラクタ。
         */
        ~ScopedSession() NN_NOEXCEPT;
    };

    /**
     * @brief       デバッグ支援機能の利用開始宣言及び初期化。
     *
     * @param[out]  pOutSession     デバッグ支援機能有効期間管理用セッションハンドル。
     *
     * @return      処理結果が返されます。@n
     *              下記列挙された返値以外の場合は対象機能を利用許可がないか、システム側の機能リソースが不足しているため利用できない状態です。
     *
     * @retval      nn::ResultSuccess                           正常に処理が終了しました。
     * @retval      nn::ec::ResultShopServiceAccessOverRequest  既に初期化済の状態で呼び出されました。
     *
     * @pre         pOutSession != nullptr
     *
     * @details     本クラスの各支援機能の利用は、本初期化が行われなければ利用できません。@n
     *              デバッグ支援機能のリソースコンテキストははシングルトンコンテキストモデルです。@n
     *              全てのセッションハンドルが解放された時にコンテキストも終了解放されます。
     */
    static Result Initialize(ScopedSession* pOutSession) NN_NOEXCEPT;

    /**
     * @brief       デバッグ支援機能の利用可否状態の更新要求。
     *
     * @details     デバッグ支援機能を利用するには以下のファームウェアデバッグ機能を有効に指定する必要があります。@n
     *              ファームウェアデバッグ設定を変更後、本関数を呼び出す事で、再起動なしにデバッグ支援機能の有効状態を反映できます。@n
     *              ※再起動による反映でも問題ありません。@n
     *
     *              @li "nim.errorsimulate", "enable_error_simulate"@n
     *                  true:   デバッグ支援機能を有効にします。@n
     *                  false:  デバッグ支援機能を無効にします。@n
     *
     * @note        本インタフェースを利用するには、@ref DebugForShopServiceAccessor::Initialize() を呼び出している必要があります。
     */
    static void RefreshAvailability() NN_NOEXCEPT;

    /**
     * @brief       登録済デバッグレスポンスの全消去要求。(デバッグレスポンス支援機能カテゴリ)
     *
     * @return      処理結果が返されます。@n
     *              下記列挙された返値以外の場合は対象機能を利用許可がないか、システム側の機能リソースが不足しているため利用できない状態です。
     *
     * @retval      nn::ResultSuccess       正常に処理が終了しました。
     *
     * @details     登録中の全デバッグレスポンスを消去します。
     *
     * @note        本インタフェースを利用するには、@ref DebugForShopServiceAccessor::Initialize() を呼び出している必要があります。
     */
    static Result ClearResponse() NN_NOEXCEPT;

    /**
     * @brief       デバッグレスポンス登録要求。(デバッグレスポンス支援機能カテゴリ)
     *
     * @param[in]   keyTarget       照合対象キー[接続先ショップサービスサーバー種別]。
     * @param[in]   pKeyPath        照合対象キー[URLパスキー文字列]。@n
     *                              登録元の文字列に null 終端文字が必要です。
     * @param[in]   pValue          照合適用時採用レスポンスボディデータ。
     * @param[in]   valueSize       照合適用時採用レスポンスボディデータ長( バイトサイズ )。
     * @param[in]   expectResult    照合適用時採用Result値。
     * @param[in]   happenedRate    デバッグレスポンス適用事象発生率( 10000分率, 10000 で 100% 発生します。 )。
     *
     * @return      処理結果が返されます。@n
     *              下記列挙された返値以外の場合は対象機能を利用許可がないか、システム側の機能リソースが不足しているため利用できない状態です。
     *
     * @retval      nn::ResultSuccess                               正常に処理が終了しました。
     * @retval      nn::ec::ResultShopServiceAccessInvalidCharacter 要求パラメータ中にパラメータとして使用不可の無効文字を検出しました。
     *
     * @details     指定のキー内容に対するレスポンスボディを値としたペアデータとしてデバッグレスポンスに登録します。
     *
     * @note        本インタフェースを利用するには、@ref DebugForShopServiceAccessor::Initialize() を呼び出している必要があります。
     */
    static Result RegisterResponse(const ShopService& keyTarget, const char* pKeyPath, const char *pValue, const size_t valueSize, const Result& expectResult, const uint32_t happenedRate) NN_NOEXCEPT;
};

}}

